/*******************************************************************************
 * Copyright (c) 2007, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - initial selection handling improvements
 *******************************************************************************/
package org.simantics.modeling.ui.pdf;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPersistentPreferenceStore;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.IExportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.preferences.ScopedPreferenceStore;
import org.simantics.NameLabelMode;
import org.simantics.NameLabelUtil;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.requests.Node;
import org.simantics.modeling.ui.Activator;
import org.simantics.modeling.ui.utils.NoProjectPage;
import org.simantics.project.IProject;
import org.simantics.project.ProjectKeys;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.ui.utils.ResourceAdaptionUtils;
import org.simantics.utils.FileUtils;
import org.simantics.utils.strings.AlphanumComparator;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PDFDiagramExportWizard extends Wizard implements IExportWizard {

    private static final Logger LOGGER = LoggerFactory.getLogger(PDFDiagramExportWizard.class);

    private static final int    MAX_RECENT_EXPORT_PATHS = 10;

    PDFExportPlan               exportPlan;

    private boolean readPreferences(PDFExportPlan toPlan) {
        IPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);
        store.setDefault(Preferences.DIAGRAM_EXPORT_PDF_INCLUDE_DIAGRAMS, true);
        store.setDefault(Preferences.DIAGRAM_EXPORT_PDF_INCLUDE_DOCUMENTATION, true);
        store.setDefault(Preferences.DIAGRAM_EXPORT_PDF_INCLUDE_COMPONENT_LEVEL_DOCUMENTATION, true);
        store.setDefault(Preferences.DIAGRAM_EXPORT_PDF_ADD_PAGE_NUMBERS, true);

        toPlan.recentLocations = Preferences.decodePaths(store.getString(Preferences.DIAGRAM_EXPORT_PDF_PATH));
        toPlan.fitContentToPageMargins = store.getBoolean(Preferences.DIAGRAM_EXPORT_PDF_ZOOM_TO_FIT);
        toPlan.attachTG = store.getBoolean(Preferences.DIAGRAM_EXPORT_PDF_ATTACH_TG);
        toPlan.includeDiagrams = store.getBoolean(Preferences.DIAGRAM_EXPORT_PDF_INCLUDE_DIAGRAMS);
        toPlan.includeDocumentation = store.getBoolean(Preferences.DIAGRAM_EXPORT_PDF_INCLUDE_DOCUMENTATION);
        toPlan.includeComponentLevelDocumentation = store.getBoolean(Preferences.DIAGRAM_EXPORT_PDF_INCLUDE_COMPONENT_LEVEL_DOCUMENTATION);
        toPlan.addPageNumbers = store.getBoolean(Preferences.DIAGRAM_EXPORT_PDF_ADD_PAGE_NUMBERS);

        return true;
    }

    private void writePreferences(PDFExportPlan plan) throws IOException {
        IPersistentPreferenceStore store = new ScopedPreferenceStore(InstanceScope.INSTANCE, Activator.PLUGIN_ID);

        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_PATH, Preferences.encodePaths(plan.recentLocations));
        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_ZOOM_TO_FIT, String.valueOf(plan.fitContentToPageMargins));
        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_ATTACH_TG, String.valueOf(plan.attachTG));
        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_INCLUDE_DIAGRAMS, String.valueOf(plan.includeDiagrams));
        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_INCLUDE_DOCUMENTATION, String.valueOf(plan.includeDocumentation));
        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_INCLUDE_COMPONENT_LEVEL_DOCUMENTATION, String.valueOf(plan.includeComponentLevelDocumentation));
        store.putValue(Preferences.DIAGRAM_EXPORT_PDF_ADD_PAGE_NUMBERS, String.valueOf(plan.addPageNumbers));

        if (store.needsSaving())
            store.save();
    }

    public PDFDiagramExportWizard() {
        setWindowTitle("Export Model to PDF");
        setNeedsProgressMonitor(true);
    }

    @Override
    public void addPages() {
        super.addPages();
        if (exportPlan != null) {
            addPage(new PDFExportPage(exportPlan));
        } else {
            addPage(new NoProjectPage("Export Model to PDF"));
        }
    }

    private NamedResource toNamedResource(ReadGraph graph, Resource r) throws DatabaseException {
        String name = NameLabelUtil.modalName(graph, r, NameLabelMode.NAME_AND_LABEL);
        return new NamedResource(name, r);
    }

    @Override
    public void init(IWorkbench workbench, IStructuredSelection selection) {
        ISessionContext ctx = Simantics.getSessionContext();
        if (ctx == null)
            return;
        IProject project = ctx.getHint(ProjectKeys.KEY_PROJECT);
        if (project == null)
            return;

        exportPlan = new PDFExportPlan(ctx);
        exportPlan.project = project;
        exportPlan.initialSelection = selection;
        readPreferences(exportPlan);

        // Get all model names
        try {
            exportPlan.sessionContext.getSession().syncRequest(new ReadRequest() {
                @Override
                public void run(ReadGraph graph) throws DatabaseException {
                    Set<Resource> processed = new HashSet<>();
                    List<NamedResource> models = new ArrayList<>();

                    Collection<Resource> ontologies = Simantics.applySCL("Simantics/SharedOntologies", "traverseSharedOntologies", graph, graph.getRootLibrary());
                    for (Resource root : ontologies) {
                        if (processed.add(root))
                            models.add( toNamedResource(graph, root) );
                    }

                    for (Resource model : graph.syncRequest(new ObjectsWithType(exportPlan.project.get(),
                            Layer0.getInstance(graph).ConsistsOf, SimulationResource.getInstance(graph).Model))) {
                        if (processed.add(model))
                            models.add( toNamedResource(graph, model) );
                    }
                    Collections.sort(models, AlphanumComparator.CASE_INSENSITIVE_COMPARATOR);
                    exportPlan.selectableModels.addAll(models);

                    Resource selected = ResourceAdaptionUtils.toSingleResource(selection.getFirstElement());
                    Resource indexRoot = selected != null ? graph.sync(new PossibleIndexRoot(selected)) : null;
                    if (indexRoot != null)
                        exportPlan.initialModelSelection = exportPlan.selection = toNamedResource(graph, indexRoot);

                    if (exportPlan.selection == null && !exportPlan.selectableModels.isEmpty())
                        exportPlan.selection = exportPlan.selectableModels.get(0);
                }
            });
        } catch (DatabaseException e) {
            LOGGER.error("Failed to initialize diagram PDF export wizard input data.", e);
        }
    }

    @Override
    public boolean performFinish() {
        Path target = exportPlan.exportLocation.toPath();
        if (Files.exists(target, LinkOption.NOFOLLOW_LINKS)) {
            boolean confirmed = MessageDialog.openConfirm(getShell(), "Overwrite", "Are you sure you want to overwrite " + exportPlan.exportLocation);
            if (!confirmed)
                return false;

            try {
                FileUtils.delete(target);
            } catch (IOException e) {
                ExceptionUtils.logAndShowError(e);
                return false;
            }
        }

        Preferences.addFirstRemovingDuplicates(exportPlan.recentLocations, exportPlan.exportLocation.getAbsolutePath(), MAX_RECENT_EXPORT_PATHS);

        try {
            writePreferences(exportPlan);
        } catch (IOException e) {
            ErrorLogger.defaultLogError("Failed to write preferences", e);
        }

        // Make sure that the diagrams are printed in the same order as the user
        // saw them in the wizard.
        exportPlan.selectedNodes = exportPlan.nodes.depthFirstFlatten(
                exportPlan.selectedNodeSet::contains,
                Node.CASE_INSENSITIVE_COMPARATOR);

        long start = System.currentTimeMillis();
        try {
            getContainer().run(true, true, monitor -> {
                try {
                    DiagramPrinter.printToPdf(monitor, exportPlan, exportPlan.exportLocation.toString(), exportPlan.selectedNodes);
                } catch (PdfException e) {
                    throw new InvocationTargetException(e);
                } finally {
                    monitor.done();
                }
            });
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            ExceptionUtils.logAndShowError(t);
            return false;
        } catch (InterruptedException e) {
            return false;
        }
        long end = System.currentTimeMillis();
        LOGGER.info("PDF export took " + ((end - start) * 1e-3) + " seconds.");

        return true;
    }

}
