/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagramEditor.handlers.e4;

import javax.inject.Inject;
import javax.inject.Named;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.e4.core.contexts.Active;
import org.eclipse.e4.core.di.annotations.CanExecute;
import org.eclipse.e4.core.di.annotations.Execute;
import org.eclipse.e4.core.di.annotations.Optional;
import org.eclipse.e4.ui.model.application.ui.basic.MPart;
import org.eclipse.e4.ui.model.application.ui.basic.MWindow;
import org.eclipse.e4.ui.model.application.ui.menu.MToolItem;
import org.eclipse.e4.ui.services.IServiceConstants;
import org.eclipse.e4.ui.workbench.modeling.EModelService;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.internal.e4.compatibility.CompatibilityEditor;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.layers.ILayersEditor;
import org.simantics.modeling.ui.diagramEditor.DiagramEditor;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * @author Tuukka Lehtonen
 */
@SuppressWarnings("restriction")
public class ToggleFocusabilityHandler {

	private static final String HANDLED_ITEM_ID = "org.simantics.modeling.ui.focusAllToggle";

	@Inject
	private EModelService modelService;

	private void updateStateForPart(MPart part) {
		IWorkbenchPart wbPart = tryGetWorkbenchPart(part);
		if (wbPart != null) {
			MWindow win = modelService.getTopLevelWindowFor(part);
			MToolItem item = (MToolItem) modelService.find(HANDLED_ITEM_ID, win);
			ILayersEditor le = getLayers(wbPart);
			if (item != null && le != null) {
				setToolItemState(item, le.getFocusImages());
			}
		}
	}

	// tracks the active part
	@Inject
	@Optional
	public void receiveActivePart(@Named(IServiceConstants.ACTIVE_PART) MPart activePart) {
		updateStateForPart(activePart);
	}

	@CanExecute
	public boolean canExecute(@Active MPart part) {
		return tryGetWorkbenchPart(part) instanceof DiagramEditor;
	}

	@Execute
	public void execute(@Optional MToolItem toolItem) {
		ILayersEditor le = getLayers();
		if (le != null) {
			boolean newValue = !le.getFocusImages();
			le.setFocusImages( newValue );
			if (toolItem != null) {
				setToolItemState(toolItem, newValue);
			}
		}
	}

	private void setToolItemState(MToolItem item, boolean focusImages) {
		item.setSelected(focusImages);
		item.setTooltip((focusImages ? "Deny" : "Allow") + " Focusing and Editing of Images");
	}

	protected ILayersEditor getLayers() {
		DiagramEditor editor = getEditor();
		if (editor == null)
			return null;
		return getLayers(editor);
	}

	protected ILayersEditor getLayers(IAdaptable editor) {
		// The diagram might not be available since the diagram editor loads it asynchronously.
		IDiagram diagram = (IDiagram) editor.getAdapter(IDiagram.class);
		if (diagram == null)
			return null;
		//System.out.println("getLayersEditor(" + diagram + ")");
		ILayersEditor le = diagram.getHint(DiagramHints.KEY_LAYERS_EDITOR);
		return le;
	}

	protected DiagramEditor getEditor() {
		IEditorPart editorPart = WorkbenchUtils.getActiveEditor();
		if (editorPart == null)
			return null;
		if (editorPart instanceof DiagramEditor) {
			DiagramEditor editor = (DiagramEditor) editorPart;
			return editor;
		}
		return null;
	}

	private IWorkbenchPart tryGetWorkbenchPart(MPart part) {
		if (part == null)
			return null;
		Object obj = part.getObject();
		if (obj instanceof CompatibilityEditor) {
			CompatibilityEditor editor = (CompatibilityEditor) obj;
			return editor.getPart();
		} else if (obj instanceof IWorkbenchPart) {
			return (IWorkbenchPart) obj; 
		}
		return null;
	}

}
