/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - Decoration positioning improvements
 *******************************************************************************/
package org.simantics.modeling.ui.diagram.style;

import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.diagram.elements.DecorationSVGNode;
import org.simantics.diagram.elements.SVGNode;
import org.simantics.diagram.profile.StyleBase;
import org.simantics.document.DocumentResource;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ui.Activator;
import org.simantics.modeling.ui.diagram.style.DocumentDecorationStyle.DocumentResult;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.profile.EvaluationContext;
import org.simantics.scenegraph.profile.common.ProfileVariables;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 */
public class DocumentDecorationStyle extends StyleBase<DocumentResult> {

    /**
     * This is needed to keep the issue decoration up-to-date when its parent
     * element moves.
     */
    public static record DocumentResult(Object identier) {}

    private static final String DECORATION_NODE_NAME = "documentDecorations"; //$NON-NLS-1$

    private List<Resource> getContexts(ReadGraph graph, Resource element) throws DatabaseException {
        ModelingResources MOD = ModelingResources.getInstance(graph);
        var result = new ArrayList<Resource>(3);
        result.add(element);
        Resource config = graph.getPossibleObject(element, MOD.ElementToComponent);
        if (config != null && result.indexOf(config) == -1) result.add(config);
        config = graph.getPossibleObject(element, MOD.DiagramConnectionToConnection);
        if (config != null && result.indexOf(config) == -1) result.add(config);
        return result;
    }

    @Override
    public DocumentResult calculateStyle(ReadGraph graph, Resource runtimeDiagram, Resource entry, Resource element, Variable configuration) throws DatabaseException {
        DocumentResource DOC = DocumentResource.getInstance(graph);

        for(Resource r : getContexts(graph, element)) {
            if(graph.hasStatement(r, DOC.HasDocumentation)) {
                return new DocumentResult(DecorationStyles.getStyleIdentifier(graph, runtimeDiagram, element));
            }
        }

        return null;
    }

    @Override
    public void applyStyleForNode(EvaluationContext observer, INode node, DocumentResult result) {
        if (result == null) {
            ProfileVariables.denyChild(node, "", DECORATION_NODE_NAME); //$NON-NLS-1$
            return;
        }

        SVGNode svgNode = ProfileVariables.claimChild(node, "", DECORATION_NODE_NAME, DecorationSVGNode.class, observer); //$NON-NLS-1$

        svgNode.setZIndex( Integer.MAX_VALUE );
        svgNode.setTransform(DecorationStyles.getDecorationPosition(node, boundsToTransform, connectionOffset));
        svgNode.setData(Activator.DOCUMENT_SVG_TEXT);
    }

    private static Point2D connectionOffset = new Point2D.Double(-2, 2);

    private static Function<Rectangle2D, AffineTransform> boundsToTransform = bounds -> {
        return AffineTransform.getTranslateInstance(bounds.getMinX() - 1.2, bounds.getMaxY() + 1.2);
    };

    @Override
    protected void cleanupStyleForNode(INode node) {
        ProfileVariables.denyChild(node, "", DECORATION_NODE_NAME); //$NON-NLS-1$
    }

    @Override
    public String toString() {
        return "Document decoration"; //$NON-NLS-1$
    }

}
