/*******************************************************************************
 * Copyright (c) 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagram.style;

import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Collection;
import java.util.HashSet;
import java.util.function.Function;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.connection.RouteGraph;
import org.simantics.diagram.handler.Paster;
import org.simantics.diagram.handler.Paster.RouteLine;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.nodes.ConnectionNode;
import org.simantics.scenegraph.g2d.nodes.Decoration;
import org.simantics.scenegraph.g2d.nodes.connection.RouteGraphNode;
import org.simantics.scenegraph.utils.NodeUtil;
import org.simantics.structural.stubs.StructuralResource2;

/**
 * @author Jussi Koskela
 * @author Tuukka Lehtone 
 * @since 1.60.0 
 */
public class DecorationStyles {

	public static Object getStyleIdentifier(ReadGraph graph, Resource runtimeDiagram, Resource element) throws DatabaseException {
		DiagramResource DIA = DiagramResource.getInstance(graph);
		StructuralResource2 STR = StructuralResource2.getInstance(graph);
		if (graph.isInstanceOf(element, DIA.RouteGraphConnection)) {
			Collection<Resource> connectors = graph.getObjects(element, DIA.HasConnector);
			Collection<Resource> routeNodes = graph.getObjects(element, DIA.HasInteriorRouteNode);

			// This is needed to make this query result change every time the underlying element changes visually.
			var identifier = new HashSet<Object>(connectors.size() + routeNodes.size());

			for (Resource connector : connectors) {
				for (Resource connectedTo : graph.getObjects(connector, STR.Connects)) {
					if (!connectedTo.equals(element)) {
						AffineTransform at = DiagramGraphUtil.getDynamicAffineTransform(graph, runtimeDiagram, connectedTo, DIA.HasDynamicTransform, false);
						identifier.add(at);
					}
				}
			}
			for (Resource routeLine : routeNodes) {
				RouteLine rl = Paster.readRouteLine(graph, routeLine);
				identifier.add(rl);
			}
			return identifier;
		} else {
			return DiagramGraphUtil.getAffineTransform(graph, element);
		}
	}

	@SuppressWarnings("rawtypes")
	private static final Class[] IGNORED_NODES_FOR_POSITION = { Decoration.class, org.simantics.diagram.elements.TextNode.class };

	/**
	 * Returns position of the decoration.
	 * By default decoration is placed to the top left corner.  Override this method to change the position.
	 *  
	 * @param node
	 * @return
	 */
	public static AffineTransform getDecorationPosition(INode node, Function<Rectangle2D, AffineTransform> fromBoundsFunction, Point2D connectionOffset) {
		Rectangle2D bounds = NodeUtil.getLocalBounds(node, IGNORED_NODES_FOR_POSITION);
		if (bounds == null) {
			return AffineTransform.getTranslateInstance(0, 0);
		}

		if (node instanceof ConnectionNode cn) {
			for (INode child : cn.getSortedNodes()) {
				if (child instanceof RouteGraphNode) {
					RouteGraphNode rgn = (RouteGraphNode) child;
					RouteGraph rg = rgn.getRouteGraph();
					Point2D nearest = rg.findNearestPoint(bounds.getCenterX(), bounds.getCenterY());
					if (nearest != null) {
						double ox = 0, oy = 0;
						if (connectionOffset != null) {
							ox = connectionOffset.getX();
							oy = connectionOffset.getY();
						}
						return AffineTransform.getTranslateInstance(nearest.getX()+ox, nearest.getY()+oy);
					}
				}
			}
		}

		return fromBoundsFunction.apply(bounds);
	}

}