package org.simantics.modeling.validation;

import java.util.Collection;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.structural.stubs.StructuralResource2;

public class ValidateMapping extends ReadRequest {

	Resource composite;
	Resource diagram;
	
	ReadGraph g;
	Layer0 b;
	DiagramResource dr;
	StructuralResource2 sr;
	ModelingResources mr;	
	
	public ValidateMapping(Resource composite) {
		this.composite = composite;
	}

	@Override
	public void run(ReadGraph g) throws DatabaseException {
		this.g = g;
		this.b = Layer0.getInstance(g);
		this.dr = DiagramResource.getInstance(g);
		this.sr = StructuralResource2.getInstance(g);
		this.mr = ModelingResources.getInstance(g);
	
		this.diagram = g.getPossibleObject(composite, mr.CompositeToDiagram);
		
		System.out.println("------------------------------------------------------------------");
		System.out.println("---- " + NameUtils.getSafeName(g, composite));
		if (diagram == null) {
			System.out.println("No diagram.");
		} else {
			additiveRule();
			destructiveRule();
		}
	}

	private void additiveRule() throws DatabaseException {
		for(Resource element : OrderedSetUtils.toList(g, diagram)) {
			Resource elementType = g.getPossibleObject(element, b.InstanceOf);
			if(elementType == null) {
				System.out.println("Element " + NameUtils.getSafeName(g, element) + " has no type.");
				continue;
			}
			Resource componentType = g.getPossibleObject(elementType, mr.SymbolToComponentType);
			if(componentType != null)
				createComponentRule(element, componentType);
			else if(g.isInstanceOf(element, dr.Connection))
				createNormalConnectionRule(element);
			else if(g.isInstanceOf(element, dr.Flag))
				createFlagRule(element);
			/*else
				System.out.println("No rule for element " + NameUtils.getSafeName(g, element));
				*/
		}
	}		

	private void createComponentRule(Resource element, Resource componentType) throws DatabaseException {
		Collection<Resource> components = g.getObjects(element, mr.ElementToComponent);
		if(components.isEmpty())
			System.out.println("Element " + NameUtils.getSafeName(g, element) + " is not mapped.");
		else if(components.size() > 1)
			System.out.println("Element " + NameUtils.getSafeName(g, element) + " is not multiple times.");
		else for(Resource component : components) {
			if(!g.isInstanceOf(component, componentType))
				System.out.println("Component " + NameUtils.getSafeName(g, component) + 
						" is not an instance of " + NameUtils.getSafeName(g, componentType));
			if(!g.hasStatement(component, b.PartOf, composite))
				System.out.println("Component " + NameUtils.getSafeName(g, component) + " is not a part of composite.");
			if(!g.hasStatement(component, mr.Mapped))
				System.out.println("Component " + NameUtils.getSafeName(g, component) + " is not tagged as Mapped.");
		}
	}

	private void createNormalConnectionRule(Resource element) throws DatabaseException {
		Collection<Resource> connections = g.getObjects(element, mr.DiagramConnectionToConnection);
		if(connections.isEmpty())
			System.out.println("Diagram connection " + NameUtils.getSafeName(g, element) + " is not mapped.");
		else if(connections.size() > 1)
			System.out.println("Diagram connection " + NameUtils.getSafeName(g, element) + " is not multiple times.");
		else for(Resource connection : connections) {
			if(!g.isInstanceOf(connection, sr.Connection))
				System.out.println("Resource " + NameUtils.getSafeName(g, connection) +	" is not a connection.");
			if(!g.hasStatement(connection, mr.Mapped))
				System.out.println("Connection " + NameUtils.getSafeName(g, connection) + " is not tagged as Mapped.");
			
			for(Resource connector : g.getObjects(element, sr.IsConnectedTo)) {
				for(Statement stat : g.getStatements(connector, sr.Connects)) {
					Resource diagramConnectionRelation = g.getInverse(stat.getPredicate());
					Resource cElement = stat.getObject();
					Resource connectionRelation = g.getPossibleObject(diagramConnectionRelation, mr.DiagramConnectionRelationToConnectionRelation);
					if(connectionRelation == null)
						continue;
					Resource component = g.getPossibleObject(cElement, mr.ElementToComponent);
					if(component == null)
						continue;
					
					// Special connection data
					Resource connectionRelation2 = g.getPossibleObject(diagramConnectionRelation, mr.DiagramConnectionRelationToConnectionRelationB);
					Resource connectionRelation3 = g.getPossibleObject(diagramConnectionRelation, mr.DiagramConnectionRelationToConnectionRelationC);
					Resource componentType = g.getPossibleObject(diagramConnectionRelation, mr.DiagramConnectionRelationToComponentType);
					
					// Normal connection
					if(connectionRelation2 == null || connectionRelation3 == null || componentType == null) {
						if(!g.hasStatement(component, connectionRelation, connection))
							System.out.println("Missing connection statement " + NameUtils.getSafeName(g, component) + " . " +
									NameUtils.getSafeName(g, connectionRelation));
					}
					
					// Special connection
					else {
						Resource component2 = g.getPossibleObject(element, mr.ElementToComponent);
						if(component2 == null)
							System.out.println("Special connection " + NameUtils.getSafeName(g, element) + " is not mapped.");
						else {
							if(!g.isInstanceOf(component2, componentType))
								System.out.println("Component " + NameUtils.getSafeName(g, component2) + 
										" is not an instance of " + NameUtils.getSafeName(g, componentType));
							if(!g.hasStatement(component2, b.PartOf, composite))
								System.out.println("Component " + NameUtils.getSafeName(g, component2) + " is not a part of composite.");
							if(!g.hasStatement(component2, mr.Mapped))
								System.out.println("Component " + NameUtils.getSafeName(g, component2) + " is not tagged as Mapped.");
							
							boolean foundConnection2 = false;
							for(Resource connection2 : g.getObjects(component2, connectionRelation2))
								if(g.hasStatement(component, connectionRelation, connection2)) {
									foundConnection2 = true;
									if(!g.isInstanceOf(connection2, sr.Connection))
										System.out.println(NameUtils.getSafeName(g, connection2) + " is not a connection.");
									if(!g.hasStatement(component2, connectionRelation3, connection))
										System.out.println("Missing connection point " + NameUtils.getSafeName(g, component2) + " . "
												+ NameUtils.getSafeName(g, connectionRelation3));
								}
							if(!foundConnection2)
								System.out.println("Special connection "+ NameUtils.getSafeName(g, element) + " is not correctly mapped (missing configuration connection).");
						}
					}
				}
			}
		}
	}

	/*
                claim(          
      	            
      	            // Create a connection
      	            exists(
                        and(statement_bbf(Component2, ConnectionRelation2, Connection2),
                            statement    (Component,  ConnectionRelation,  Connection2)
                        ),
                        Connection2
                    ),
                    b(sr.Connection, Connection2),
                    //b(mapped, Connection2),
      	            
      	            // 
      	            statement(Component2, ConnectionRelation3, Connection)
                ),
}*/
	
	private void createFlagRule(Resource element) throws DatabaseException {
		for(Resource connectionJoin : g.getObjects(element, dr.FlagIsJoinedBy))
			for(Resource connector : g.getObjects(element, sr.IsConnectedTo))
				for(Resource diagramConnection : g.getObjects(connector, sr.Connects))
					for(Resource connection : g.getObjects(diagramConnection, mr.DiagramConnectionToConnection)) {
						if(!g.hasStatement(connectionJoin, sr.Joins, connection))
							System.out.println("Joins-relation of flag " + NameUtils.getSafeName(g, element) + " is not mapped.");
						if(!g.hasStatement(composite, sr.HasConnectionJoin, connectionJoin))
							System.out.println("Connection join is not attached to the composite.");
					}
		
	}
	
	private void destructiveRule() throws DatabaseException {
		for(Resource component : g.getObjects(composite, b.ConsistsOf)) {
			if(g.hasStatement(component, mr.Mapped)) {
				//Resource element = g.getPossibleObject(component, mr.ComponentToElement);
				Collection<Resource> elements = g.getObjects(component, mr.ComponentToElement);
				if (elements.isEmpty())
					System.out.println("Component " + NameUtils.getSafeName(g, component) + 
							" does not have corresponding element (but is tagged Mapped).");
				for (Resource element : elements) {
					if(!OrderedSetUtils.contains(g, diagram, element))
						System.out.println("Element corresponding to component " + NameUtils.getSafeName(g, component) + 
						" is not included in the diagram.");
					else {
						for(Statement stat : g.getStatements(component, sr.IsConnectedTo)) {
							//Resource connectionRelation = stat.getPredicate();
							Resource connection = stat.getObject();
							if(g.hasStatement(connection, mr.Mapped)) {
								if (!g.hasStatement(connection, mr.ConnectionMapsTo))
									System.out.println("Connection " + NameUtils.getSafeName(g, connection) + 
									" does not have a correspondence in diagram (but is tagged Mapped).");
							}
						}
					}
				}
			}
		}
	}

}
