package org.simantics.modeling.utils;

import java.util.Collections;
import java.util.Set;

import org.simantics.db.MetadataI;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.UndoMetadata;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.genericrelation.DependencyChanges;
import org.simantics.db.layer0.genericrelation.DependencyChanges.ComponentAddition;
import org.simantics.db.layer0.genericrelation.DependencyChanges.ComponentModification;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.service.CollectionSupport;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ModelingUtils;
import org.simantics.modeling.adapters.AddMissingIdentifiers;
import org.simantics.modeling.adapters.ChangeHistoryUpdated;
import org.simantics.modeling.adapters.SkipChangeHistoryUpdate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 */
public class OntologicalRequirementTracker {

	private static final Logger LOGGER = LoggerFactory.getLogger(OntologicalRequirementTracker.class);

	private static final boolean DEBUG = false;
	private static final boolean DEBUG_MODIFICATIONS = false;
	private static final boolean DEBUG_PROPAGATE = false;

	private Layer0 L0;
	private DiagramResource DIA;
	private boolean trace;
	private boolean debug;

	private void propagate(ReadGraph graph, Resource r, Set<Resource> rs) throws DatabaseException {
		if (!r.isPersistent()) return;
		if (!rs.add(r)) return;
		if (DEBUG_PROPAGATE && trace)
			LOGGER.trace("propagated to: {}", NameUtils.getURIOrSafeNameInternal(graph, r));
		if (graph.isInstanceOf(r, L0.IndexRoot)) return;
		Resource owner = graph.getPossibleObject(r, L0.IsOwnedBy);
		if (owner == null)
			return;
		if (graph.isInstanceOf(r, DIA.DiagramContainer)) {
			// Diagram changes do not touch the composite - but do propagate from folder
			owner = graph.getPossibleObject(owner, L0.PartOf);
			if (owner == null) return;
		}
		propagate(graph, owner, rs);
	}

	public void update(ReadGraph graph, MetadataI metadata, DependencyChanges event) throws DatabaseException {

		trace = LOGGER.isTraceEnabled();
		debug = LOGGER.isDebugEnabled();

		boolean updateChangeHistory = false;
		boolean addMissingIdentifiers = true;

		if (metadata != null) {
			var metadataMap = metadata.getMetadata();
			boolean isUndo = metadataMap.containsKey(UndoMetadata.class.getName());
			boolean wasHistoryUpdate = metadataMap.containsKey(ChangeHistoryUpdated.class.getName());
			boolean skipChangeHistoryUpdate = metadataMap.containsKey(SkipChangeHistoryUpdate.class.getName());
			addMissingIdentifiers = metadataMap.containsKey(AddMissingIdentifiers.class.getName());
			if ((isUndo || wasHistoryUpdate || skipChangeHistoryUpdate) && !addMissingIdentifiers) {
				if (DEBUG && debug)
					LOGGER.debug("skipping change history update: isUndo={}, wasHistoryUpdate={}, skipChangeHistoryUpdate={}",
							isUndo, wasHistoryUpdate, skipChangeHistoryUpdate);
				return;
			}
			updateChangeHistory = !skipChangeHistoryUpdate;
		}

		if (DEBUG && debug)
			LOGGER.debug("skipping change history update: updateChangeHistory={}, addMissingIdentifier={}",
					updateChangeHistory, addMissingIdentifiers);

		L0 = Layer0.getInstance(graph);
		DIA = DiagramResource.getInstance(graph);
		ModelingResources MOD = ModelingResources.getInstance(graph);

		CollectionSupport cs = graph.getService(CollectionSupport.class);

		Set<Resource> work = cs.createSet();
		Set<Resource> creates = cs.createSet();
		Set<Resource> ids = cs.createSet();

		for (Resource model : event.modelChanges.keySet()) {
			DependencyChanges.Change[] changes = event.modelChanges.get(model);
			if (changes == null || changes.length == 0)
				continue;

			for (DependencyChanges.Change c : changes) {
				if (updateChangeHistory) {
					if (c instanceof ComponentModification cm) {
						Resource r = cm.component;
						if (DEBUG && trace)
							LOGGER.trace("ComponentModification: {}", NameUtils.getURIOrSafeNameInternal(graph, r));
						propagate(graph, r, work);
					}
				}
				if (c instanceof ComponentAddition ca) {
					Resource r = ca.component;
					if (updateChangeHistory) {
						boolean hasChangeInformation = graph.hasStatement(r, MOD.changeInformation);
						boolean modifyChangeInformation = !hasChangeInformation && ModelingUtils.needsModificationInfo(graph, r);
						if (DEBUG && trace)
							LOGGER.trace("ComponentAddition(modifyChangeInformation={}, {})", modifyChangeInformation, NameUtils.getURIOrSafeNameInternal(graph, r));
						if (modifyChangeInformation) {
							creates.add(r);
						}
						propagate(graph, r, work);
					}

					boolean hasGUID = graph.hasStatement(r, L0.identifier);
					if (!hasGUID && ModelingUtils.needsIdentifier(graph, r)) {
						if (DEBUG && trace)
							LOGGER.trace("ComponentAddition(needs ID, {})", NameUtils.getURIOrSafeNameInternal(graph, r));
						ids.add(r);
					}
				}
			}
		}

		if (creates.isEmpty() && work.isEmpty() && ids.isEmpty())
			return;

		Set<Resource> modis = Collections.emptySet();
		if (updateChangeHistory && !work.isEmpty()) {
			modis = cs.createSet();
			for (Resource r : Layer0Utils.sortByCluster(graph, work))
				if (!creates.contains(r) && ModelingUtils.needsModificationInfo(graph, r))
					modis.add(r);
		}

		if (DEBUG_MODIFICATIONS && debug) {
			for (Resource w : creates)
				LOGGER.debug("created: {}", NameUtils.getURIOrSafeNameInternal(graph, w));
			for (Resource w : modis)
				LOGGER.debug("modified: {}", NameUtils.getURIOrSafeNameInternal(graph, w));
			for (Resource w : ids)
				LOGGER.debug("id needed: {}", NameUtils.getURIOrSafeNameInternal(graph, w));
		}

		if (!modis.isEmpty() || !creates.isEmpty() || !ids.isEmpty()) {
			graph.asyncRequest(new OntologicalRequirementEnforceRequest(creates, modis, ids));
		}

	}

}
