/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.subscription;

import java.util.Collection;
import java.util.UUID;

import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Databoard;
import org.simantics.databoard.annotations.Optional;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.util.Bean;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.preferences.SubscriptionPreferences;


public class AddSubscription extends WriteRequest {

    Resource         container;
    String           subscriptionName;
    Collection<Item> items;
    boolean          overwrite;

    public AddSubscription(Resource container, String subscriptionName, Collection<Item> items, boolean overwrite) {
        this.container = container;
        this.subscriptionName = subscriptionName; 
        this.items = items;
        this.overwrite = overwrite;
    }

    @Override
    public void perform(WriteGraph graph) throws DatabaseException {
        Layer0 l0 = Layer0.getInstance(graph);
        ModelingResources mod = ModelingResources.getInstance(graph);

        IEclipsePreferences pn = InstanceScope.INSTANCE.getNode( SubscriptionPreferences.P_NODE );
        Double defaultSubscriptionItemInterval = pn.getDouble(SubscriptionPreferences.P_SUBSCRIPTION_INTERVAL, SubscriptionPreferences.DEFAULT_SUBSCRIPTION_INTERVAL);
        Double defaultSubscriptionItemDeadband = pn.getDouble(SubscriptionPreferences.P_SUBSCRIPTION_DEADBAND, SubscriptionPreferences.DEFAULT_SUBSCRIPTION_DEADBAND); 

        if (overwrite) {
            for (Resource existing : graph.syncRequest(new ObjectsWithType(container, l0.ConsistsOf, mod.Subscription))) {
                String name = graph.getPossibleRelatedValue(existing, l0.HasName, Bindings.STRING);
                if (subscriptionName.equals(name)) {
                    RemoverUtil.remove(graph, existing);
                }
            }
        }

        Resource subscription = graph.newResource();
        graph.claim(subscription, l0.InstanceOf, null, mod.Subscription);
        String freshName = NameUtils.findFreshName(graph, subscriptionName, container);
        graph.claimLiteral(subscription, l0.HasName, freshName);
        graph.claimLiteral(subscription, l0.HasLabel, freshName);
        graph.claim(container, l0.ConsistsOf, subscription);

    	Binding rviBinding = graph.getService(Databoard.class).getBindingUnchecked( RVI.class );
        for (Item item : items) {
            Resource subscriptionItem = graph.newResource();
            graph.claim(subscriptionItem, l0.InstanceOf, null, mod.Subscription_Item);
            graph.claim(subscription, l0.ConsistsOf, subscriptionItem);
            graph.claimLiteral(subscriptionItem, l0.HasName, UUID.randomUUID().toString(), Bindings.STRING);
            if ( item.label!= null) graph.claimLiteral(subscriptionItem, l0.HasLabel, item.label, Bindings.STRING);
            
            graph.claimLiteral(subscriptionItem, mod.Subscription_Item_VariableId, l0.RVI, item.resolvedName, rviBinding);

            Double interval = item.interval!=null ? item.interval : defaultSubscriptionItemInterval;
            if (interval != null) {
                graph.claimLiteral(subscriptionItem, mod.Subscription_Item_SamplingInterval, interval, Bindings.DOUBLE);
            }
            
            Double deadband = item.deadband!=null ? item.deadband : defaultSubscriptionItemDeadband;
            if (deadband != null) {
                graph.claimLiteral(subscriptionItem, mod.Subscription_Item_Deadband, deadband, Bindings.DOUBLE);
            }

            if (item.gain != null) {
                graph.claimLiteral(subscriptionItem, mod.Subscription_Item_Gain, item.gain, Bindings.DOUBLE);
            }
            
            if (item.bias != null) {
                graph.claimLiteral(subscriptionItem, mod.Subscription_Item_Deadband, item.bias, Bindings.DOUBLE);
            }                
            
            if (item.datatype != null)
                graph.claimLiteral(subscriptionItem, mod.Subscription_Item_Datatype,
                        l0.DataType, item.datatype,
                        Bindings.getBindingUnchecked(Datatype.class));
        }
    }

    public static class Item extends Bean {
    	public RVI resolvedName;
        public @Optional Double interval, deadband, gain, bias;
        public @Optional String label;
        public @Optional Datatype datatype;
    }

}
