/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.mapping;

import java.util.Map;

import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.diagram.synchronization.ISynchronizationContext;
import org.simantics.diagram.synchronization.SynchronizationException;
import org.simantics.diagram.synchronization.SynchronizationHints;
import org.simantics.diagram.synchronization.graph.BasicResources;
import org.simantics.diagram.synchronization.graph.CopyAdvisorUtil;
import org.simantics.diagram.synchronization.graph.GraphCopyAdvisor;
import org.simantics.diagram.synchronization.graph.GraphSynchronizationHints;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ComponentUtils;
import org.simantics.modeling.ModelingUtils;
import org.simantics.modeling.services.ComponentNamingUtil;
import org.simantics.modeling.services.NamingException;
import org.simantics.project.IProject;

import gnu.trove.map.hash.THashMap;

/**
 * @author Tuukka Lehtonen
 */
public class ComponentCopyAdvisor extends GraphCopyAdvisor {

    @Override
    public Evaluation canCopy(ISynchronizationContext context, WriteGraph graph, Resource source,
            Resource sourceContainer, Resource targetContainer) throws DatabaseException {
        BasicResources br = context.get(GraphSynchronizationHints.BASIC_RESOURCES);
        return (graph.isInstanceOf(source, br.STR.Component) || graph.isInstanceOf(source, br.STR.Connection))
        ? Evaluation.SUPPORTED : Evaluation.NOT_SUPPORTED;
    }

    @Override
    public Object copy(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourceContainer,
            Resource targetContainer) throws DatabaseException {
        return copy(context, graph, source, sourceContainer, targetContainer, new THashMap<Object, Object>());
    }

    @Override
    public Object copy(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourceContainer,
            Resource targetContainer, Map<Object, Object> map) throws DatabaseException {
        Resource copy = CopyAdvisorUtil.copy2(graph, source, null, map);

        Layer0 L0 = Layer0.getInstance(graph);
        if (graph.hasStatement(sourceContainer, L0.ConsistsOf, source))
            graph.claim(targetContainer, L0.ConsistsOf, copy);

        if (ModelingUtils.needsIdentifier(graph, source))
            Layer0Utils.claimNewIdentifier(graph, copy, false);
        if (context.get(SynchronizationHints.NO_RENAME) == null)
            renameComponent(context, graph, source, copy, sourceContainer, targetContainer);
        return copy;
    }

    public static String renameComponent(ISynchronizationContext context, WriteGraph graph, Resource source,
            Resource copy, Resource sourceContainer, Resource targetContainer) throws DatabaseException {
        return renameComponent(context, graph, source, copy, sourceContainer, targetContainer,
                Layer0.getInstance(graph).HasName);
    }

    public static String renameComponent(ISynchronizationContext context, WriteGraph graph, Resource source,
            Resource copy, Resource sourceContainer, Resource targetContainer, Resource nameProperty)
            throws DatabaseException {
        BasicResources br = context.get(GraphSynchronizationHints.BASIC_RESOURCES);

        Resource sourceComponentType = graph.getPossibleType(source, br.STR.Component);
        if (sourceComponentType == null)
            return null;

        // Try to obtain a ComponentNamingStrategy for renaming the copied object.
        IProject project = context.get(SynchronizationHints.PROJECT);
        if (project == null)
            project = Simantics.getProject();

        // Try to give a valid or at least unique name for the new component.
        //Resource configurationRoot = ComponentUtils.getComponentConfigurationRoot(graph, copy);
        //Resource composite = ComponentUtils.tryGetComponentContainer(graph, copy);
        Resource configurationRoot = ComponentUtils.getCompositeConfigurationRoot(graph, targetContainer);
        Resource composite = targetContainer;

        if (configurationRoot != null) {
            try {
                String name = ComponentNamingUtil.findFreshInstanceName(graph, project, configurationRoot, composite, sourceComponentType);
                graph.claimLiteral(copy, nameProperty, name, Bindings.STRING);
                return name;
            } catch (NamingException e) {
                // This will produce duplicate names in the model,
                // should not happen if we don't run out of names.
                throw new SynchronizationException(e);
            }
        }
        return null;
    }

    @Override
    public Object cut(ISynchronizationContext context, WriteGraph graph, Resource source, Resource sourceContainer,
            Resource targetContainer) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);

        if (sourceContainer.equals(targetContainer))
            return null;

        // This handles e.g. connections, which are not part of container
        if(graph.hasStatement(sourceContainer, L0.ConsistsOf, source)) {
            graph.deny(sourceContainer, L0.ConsistsOf, source);
            graph.claim(targetContainer, L0.ConsistsOf, source);
        }

        return source;
    }

}
