/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.SelectionHints;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.utils.datastructures.hints.HintContext;

/**
 * @author Tuukka Lehtonen
 */
public class PropertyVariablesImpl extends HintContext implements PropertyVariables, IAdaptable {

    private final Variable container;
    private final Variable configuration;
    private final Variable visualized;
    private final Variable modified;
    private final String   suffix;

    public PropertyVariablesImpl(Variable container, Variable visualized, Variable modified) {
        this(container, null, visualized, modified, null);
    }

    public PropertyVariablesImpl(Variable container, Variable visualized, Variable modified, String suffix) {
        this(container, null, visualized, modified, suffix);
    }

    public PropertyVariablesImpl(Variable container, Variable configuration, Variable visualized, Variable modified) {
        this(container, configuration, visualized, modified, null);
    }

    public PropertyVariablesImpl(Variable container, Variable configuration, Variable visualized, Variable modified, String suffix) {
        if (visualized == null)
            throw new NullPointerException("null visualized variable");
        this.container = container;
        this.configuration = configuration;
        this.visualized = visualized;
        this.modified = modified;
        this.suffix = suffix;
        setHintWithoutNotification(SelectionHints.KEY_MAIN, visualized);
        setHintWithoutNotification(SelectionHints.KEY_SELECTION_PROPERTY, visualized);
    }

    @Override
    public Variable getContainer() {
        return container;
    }

    public Variable getConfiguration() {
        return configuration;
    }

    @Override
    public Variable getVisualVariable() {
        return visualized;
    }

    @Override
    public Variable getModificationVariable() {
        return modified;
    }

    @Override
    public String getSuffix() {
        return suffix;
    }

    @Override
    public PropertyVariables withSuffix(String suffix) {
        return new PropertyVariablesImpl(container, visualized, modified, suffix);
    }

    @Override
    public PropertyVariables resolved(ReadGraph graph) throws DatabaseException {
        if (suffix == null)
            return this;

        Variable c = configuration;
        Variable v = visualized;
        Variable m = modified;
        if (c != null)
            c = c.browsePossible(graph, suffix);
        if (v != null)
            v = v.browsePossible(graph, suffix);
        if (v == null)
            return null;
        if (m != null)
            m = m.browsePossible(graph, suffix);
        return new PropertyVariablesImpl(container, c, v, m, null);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((container == null) ? 0 : container.hashCode());
        result = prime * result + ((configuration == null) ? 0 : configuration.hashCode());
        result = prime * result + ((modified == null) ? 0 : modified.hashCode());
        result = prime * result + ((visualized == null) ? 0 : visualized.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        PropertyVariablesImpl other = (PropertyVariablesImpl) obj;
        if (container == null) {
            if (other.container != null)
                return false;
        } else if (!container.equals(other.container))
            return false;
        if (configuration == null) {
            if (other.configuration != null)
                return false;
        } else if (!configuration.equals(other.configuration))
            return false;
        if (modified == null) {
            if (other.modified != null)
                return false;
        } else if (!modified.equals(other.modified))
            return false;
        if (visualized == null) {
            if (other.visualized != null)
                return false;
        } else if (!visualized.equals(other.visualized))
            return false;
        return true;
    }

    @SuppressWarnings("rawtypes")
    @Override
    public Object getAdapter(Class adapter) {
        if (adapter == Variable.class)
            return visualized;
        return null;
    }

    public static List<PropertyVariables> resolve(ReadGraph graph, List<PropertyVariables> vars) throws DatabaseException {
        List<PropertyVariables> result = new ArrayList<PropertyVariables>(vars.size());
        for (PropertyVariables var : vars) {
            PropertyVariables v = var.resolved(graph);
            if (v != null)
                result.add(v);
        }
        return result;
    }

}
