/*******************************************************************************
 * Copyright (c) 2007, 2011, 2014 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties.xyline;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.browsing.ui.swt.widgets.Button;
import org.simantics.browsing.ui.swt.widgets.StringPropertyFactory;
import org.simantics.browsing.ui.swt.widgets.StringPropertyModifier;
import org.simantics.browsing.ui.swt.widgets.TrackedCombo;
import org.simantics.browsing.ui.swt.widgets.TrackedText;
import org.simantics.browsing.ui.swt.widgets.impl.ComboModifyListenerImpl;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactoryImpl;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupportImpl;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.jfreechart.chart.properties.AdjustableTab;
import org.simantics.jfreechart.chart.properties.AxisHidePropertyComposite;
import org.simantics.jfreechart.chart.properties.BooleanPropertyFactory;
import org.simantics.jfreechart.chart.properties.BooleanSelectionListener;
import org.simantics.jfreechart.chart.properties.DoubleValidator;
import org.simantics.jfreechart.chart.properties.JFreeChartPropertyColorProvider;
import org.simantics.jfreechart.chart.properties.RVIFactory;
import org.simantics.jfreechart.chart.properties.RVIModifier;
import org.simantics.jfreechart.chart.properties.TitleFactory;
import org.simantics.jfreechart.chart.properties.TitleModifier;
import org.simantics.jfreechart.chart.properties.VariableExistsValidator;
import org.simantics.layer0.Layer0;
import org.simantics.layer0.utils.direct.GraphUtils;
import org.simantics.modeling.ui.chart.property.DoublePropertyFactory;
import org.simantics.modeling.ui.chart.property.DoublePropertyModifier;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.ui.AdaptionUtils;

/**
 * PropertyTab displaying general properties and x-axis properties of a chart
 * 
 * @author Teemu Lempinen
 * @author Tuomas Miettinen
 *
 */
public class XYLineGeneralPropertiesTab extends AdjustableTab implements Widget {

    public XYLineGeneralPropertiesTab(Object id) {
        super(id);
    }

    private ScrolledComposite sc;
    private TrackedText name, title, xlabel, xvariable, xmin, xmax;
    private TrackedCombo type;
    private Button hgrid, htitle, hlegend;
    private WidgetSupportImpl domainAxisSupport = new WidgetSupportImpl();
	private Group general;
	private Label nameLabel;
	private Label labelTitle;
	private Label labelType;
	private Group hideGroup;
	private Group xgroup;
	private Label xVariableLabel;
	private Label labelMin;
	private AxisHidePropertyComposite axisHide;
	private Label labelLabel;
	private Label labelMax;
	private Composite xColumn1;
	private Composite xColumn2;

    @Override
    public void setInput(final ISessionContext context, Object input) {
        final Resource chart = AdaptionUtils.adaptToSingle(input, Resource.class);
        if(chart == null)
            return; 

        context.getSession().asyncRequest(new ReadRequest() {

            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                Layer0 l0 = Layer0.getInstance(graph);
                JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
                Resource plot = graph.syncRequest(new PossibleObjectWithType(chart, l0.ConsistsOf, jfree.Plot));
                if(plot == null) return;
                final Resource domainAxis = graph.getPossibleObject(plot, jfree.Plot_domainAxis);
                if(domainAxis == null) return;
                Display.getDefault().asyncExec(new Runnable() {
					@Override
					public void run() {
						domainAxisSupport.fireInput(context, new StructuredSelection(domainAxis));
					}
				});
                
            }
        });
    }
    
    /**
     * 
     * @author Teemu Lempinen
     *
     */
    private class TypeSelectionFactory extends ReadFactoryImpl<Resource, String> {
        @Override
        public String perform(ReadGraph graph, Resource chart) throws DatabaseException {
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);

            Resource plot = graph.syncRequest(new PossibleObjectWithType(chart, l0.ConsistsOf, jfree.XYPlot));
            if(plot != null) {
                Collection<Resource> datasets = graph.syncRequest(new ObjectsWithType(plot, l0.ConsistsOf, jfree.XYDataset));
                if(!datasets.isEmpty()) {
                    Resource dataset = datasets.iterator().next();
                    if(dataset != null) {
                        Resource renderer = graph.syncRequest(new PossibleObjectWithType(dataset, jfree.Dataset_renderer, jfree.Renderer));
                        if(renderer != null && graph.isInstanceOf(renderer, jfree.XYAreaRenderer))
                            return "Area";
                    }
                }
            }
            return "Line";
        }
    }

    /**
     * RangeItemFactory finds all inexes of a given enumeration 
     * and adds "Sum" and "All" to the returned indexes
     * @author Teemu Lempinen
     *
     */
    private class TypeItemFactory extends ReadFactoryImpl<Resource, Map<String, Object>> {
        @Override
        public Map<String, Object> perform(ReadGraph graph, Resource series) throws DatabaseException {
            LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
            result.put("Line", "Line");
            result.put("Area", "Area");
//            result.put("Stacked Area", "Stacked Area");
            return result;
        }
    }

    /**
     * TypeModifyListener for modifying the type of a bar chart 
     * @author Teemu Lempinen
     *
     */
    private class TypeModifyListener  extends ComboModifyListenerImpl<Resource> {
        @Override
        public void applyText(WriteGraph graph, Resource chart, String text) throws DatabaseException {
            JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
            Layer0 l0 = Layer0.getInstance(graph);

            Resource plot = graph.syncRequest(new PossibleObjectWithType(chart, l0.ConsistsOf, jfree.XYPlot));
            if(plot == null)
                return;

            Collection<Resource> datasets = graph.syncRequest(new ObjectsWithType(plot, l0.ConsistsOf, jfree.XYDataset));
            if(datasets == null || datasets.isEmpty())
                return;

            for(Resource dataset : datasets) {
                graph.deny(dataset, jfree.Dataset_renderer);

                Resource renderer;
                if(text.equals("Area"))
                    renderer = GraphUtils.create2(graph, jfree.XYAreaRenderer);
                else
                    renderer = GraphUtils.create2(graph, jfree.XYLineRenderer);

                graph.claim(dataset, jfree.Dataset_renderer, renderer);
            }
        }
    }

	@Override
	protected void createAndAddControls(Composite body, IWorkbenchSite site,
			ISessionContext context, WidgetSupport support) {
		support.register(this);

        // Scrolled composite containing all of the properties in this tab
        sc = new ScrolledComposite(body, SWT.NONE | SWT.H_SCROLL | SWT.V_SCROLL);
        sc.setExpandHorizontal(true);
        sc.setExpandVertical(true);

        composite = new Composite(sc, SWT.NONE);

        // General properties
        general = new Group(composite, SWT.NONE);
        general.setText("General");

        // Name
        nameLabel = new Label(general, SWT.NONE);
        nameLabel.setText("Name:");
        nameLabel.setAlignment(SWT.RIGHT);

        name = new org.simantics.browsing.ui.swt.widgets.TrackedText(general, support, SWT.BORDER);
        name.setTextFactory(new StringPropertyFactory(Layer0.URIs.HasLabel));
        name.addModifyListener(new StringPropertyModifier(context, Layer0.URIs.HasLabel));
        name.setColorProvider(new JFreeChartPropertyColorProvider(name.getResourceManager()));

        // Type
        labelType = new Label(general, SWT.NONE);
        labelType.setText("Type:");

        type = new TrackedCombo(general, support, SWT.BORDER | SWT.READ_ONLY);
        type.addModifyListener(new TypeModifyListener());
        type.setItemFactory(new TypeItemFactory());
        type.setSelectionFactory(new TypeSelectionFactory());

        // Title (Which is different than name)
        labelTitle = new Label(general, SWT.NONE);
        labelTitle.setText("Title:");

        title = new org.simantics.browsing.ui.swt.widgets.TrackedText(general, support, SWT.BORDER);
        title.setTextFactory(new TitleFactory());
        title.addModifyListener(new TitleModifier());
        title.setColorProvider(new JFreeChartPropertyColorProvider(name.getResourceManager()));

        // Group for hide options
        hideGroup = new Group(composite, SWT.NONE);
        hideGroup.setText("Hide");

        hgrid = new Button(hideGroup, support, SWT.CHECK);
        hgrid.setText("Grid");
        hgrid.setSelectionFactory(new BooleanPropertyFactory(JFreeChartResource.URIs.Plot, JFreeChartResource.URIs.Plot_visibleGrid, true));
        hgrid.addSelectionListener(new BooleanSelectionListener(context, JFreeChartResource.URIs.Plot, JFreeChartResource.URIs.Plot_visibleGrid,true));
        htitle = new Button(hideGroup, support, SWT.CHECK);
        htitle.setText("Title");
        htitle.setSelectionFactory(new BooleanPropertyFactory(JFreeChartResource.URIs.TextTitle, JFreeChartResource.URIs.visible, true));
        htitle.addSelectionListener(new BooleanSelectionListener(context, JFreeChartResource.URIs.TextTitle, JFreeChartResource.URIs.visible,true));
        hlegend = new Button(hideGroup, support, SWT.CHECK);
        hlegend.setText("Legend");
        hlegend.setSelectionFactory(new BooleanPropertyFactory(null, JFreeChartResource.URIs.Chart_visibleLegend, true));
        hlegend.addSelectionListener(new BooleanSelectionListener(context, null, JFreeChartResource.URIs.Chart_visibleLegend,true));


        // X-Axis properties
        xgroup = new Group(composite, SWT.NONE);
        xgroup.setText("X-axis");

        xColumn1 = new Composite(xgroup, SWT.NONE);
        
        // Variable for x-axis (default: empty == time)
        xVariableLabel = new Label(xColumn1, SWT.NONE);
        xVariableLabel.setText("Variable:");
        
        xvariable = new TrackedText(xColumn1, domainAxisSupport, SWT.BORDER);
        xvariable.setTextFactory(new RVIFactory());
        xvariable.addModifyListener(new RVIModifier(xvariable.getWidget(), domainAxisSupport));
        xvariable.setColorProvider(new JFreeChartPropertyColorProvider(xvariable.getResourceManager()));
        xvariable.setInputValidator(new VariableExistsValidator(support, xvariable, true));
        
        // Label for x-axis
        labelLabel = new Label(xColumn1, SWT.NONE);
        labelLabel.setText("Label:");
        
        xlabel = new TrackedText(xColumn1, domainAxisSupport, SWT.BORDER);
        xlabel.setTextFactory(new StringPropertyFactory(Layer0.URIs.HasLabel, ""));
        xlabel.addModifyListener(new StringPropertyModifier(context, Layer0.URIs.HasLabel));
        xlabel.setColorProvider(new JFreeChartPropertyColorProvider(xlabel.getResourceManager()));

        xColumn2 = new Composite(xgroup, SWT.NONE);
        
        // Min value for x-axis
        labelMin = new Label(xColumn2, SWT.NONE);
        labelMin.setText("Min:");
        labelMin.setAlignment(SWT.RIGHT);

        xmin = new TrackedText(xColumn2, domainAxisSupport, SWT.BORDER);
        xmin.setColorProvider(new JFreeChartPropertyColorProvider(xmin.getResourceManager()));
        xmin.setTextFactory(new DoublePropertyFactory(JFreeChartResource.URIs.Axis_min));
        xmin.addModifyListener(new DoublePropertyModifier(context, JFreeChartResource.URIs.Axis_min));
        xmin.setInputValidator(new DoubleValidator(true));
        
        // Max value for x-axis
        labelMax = new Label(xColumn2, SWT.NONE);
        labelMax.setText("Max:");
        
        xmax = new TrackedText(xColumn2, domainAxisSupport, SWT.BORDER);
        xmax.setColorProvider(new JFreeChartPropertyColorProvider(xmax.getResourceManager()));
        xmax.setTextFactory(new DoublePropertyFactory(JFreeChartResource.URIs.Axis_max));
        xmax.addModifyListener(new DoublePropertyModifier(context, JFreeChartResource.URIs.Axis_max));
        xmax.setInputValidator(new DoubleValidator(true));
        
        // Axis hide buttons
        axisHide = new AxisHidePropertyComposite(composite, context, domainAxisSupport, SWT.NONE);

        sc.setContent(composite);
	}

	@Override
	protected void createControlLayoutVertical() {
		// Scrolled composite containing all of the properties in this tab
        GridDataFactory.fillDefaults().grab(true, true).applyTo(sc);
        GridLayoutFactory.fillDefaults().applyTo(sc);

        GridLayoutFactory.fillDefaults().numColumns(1).margins(3, 3).applyTo(composite);

        // General properties
        GridDataFactory.fillDefaults().grab(true, false).applyTo(general);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(general);

        // Name
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(nameLabel);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(name.getWidget());

        // Type
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelType);

        GridDataFactory.fillDefaults().applyTo(type.getWidget());

        // Title (Which is different than name)
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelTitle);

        GridDataFactory.fillDefaults().span(1, 1).grab(true, false).applyTo(title.getWidget());

        // Group for hide options
        GridDataFactory.fillDefaults().applyTo(hideGroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(hideGroup);

        // X-Axis properties
        GridDataFactory.fillDefaults().grab(true, false).applyTo(xgroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(1).applyTo(xgroup);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(xColumn1);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(xColumn1);
        
        // Variable for x-axis (default: empty == time)
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(xVariableLabel);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(xvariable.getWidget());
        
        // Label for x-axis
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelLabel);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(xlabel.getWidget());

        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(xColumn2);

        // Min value for x-axis
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelMin);
        GridDataFactory.fillDefaults().applyTo(xmin.getWidget());
        
        // Max value for x-axis
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelMax);
        GridDataFactory.fillDefaults().applyTo(xmax.getWidget());
        
        // Axis hide buttons
        GridDataFactory.fillDefaults().applyTo(axisHide);

        // Set the same width to both label rows
        composite.layout();
        GridDataFactory.fillDefaults().hint(xVariableLabel.getBounds().width, SWT.DEFAULT).align(SWT.END, SWT.CENTER).applyTo(nameLabel);
        GridDataFactory.fillDefaults().hint(xVariableLabel.getBounds().width, SWT.DEFAULT).align(SWT.END, SWT.CENTER).applyTo(labelMin);
        
        Point size = composite.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        sc.setMinSize(size);
	}

	@Override
	protected void createControlLayoutHorizontal(boolean wideScreen) {
		// Scrolled composite containing all of the properties in this tab
        GridDataFactory.fillDefaults().grab(true, true).applyTo(sc);
        GridLayoutFactory.fillDefaults().applyTo(sc);

        GridLayoutFactory.fillDefaults().numColumns(2).margins(3, 3).applyTo(composite);

        // General properties
        GridDataFactory.fillDefaults().grab(true, false).applyTo(general);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(4).applyTo(general);

        // Name
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(nameLabel);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(name.getWidget());

        // Type
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelType);

        GridDataFactory.fillDefaults().applyTo(type.getWidget());

        // Title (Which is different than name)
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelTitle);

        GridDataFactory.fillDefaults().span(3, 1).grab(true, false).applyTo(title.getWidget());

        // Group for hide options
        GridDataFactory.fillDefaults().applyTo(hideGroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(hideGroup);

        // X-Axis properties
        GridDataFactory.fillDefaults().grab(true, false).applyTo(xgroup);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(xgroup);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(xColumn1);
        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(xColumn1);
        
        // Variable for x-axis (default: empty == time)
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(xVariableLabel);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(xvariable.getWidget());
        
        // Label for x-axis
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelLabel);

        GridDataFactory.fillDefaults().grab(true, false).applyTo(xlabel.getWidget());

        GridLayoutFactory.fillDefaults().margins(3, 3).numColumns(2).applyTo(xColumn2);

        // Min value for x-axis
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelMin);
        GridDataFactory.fillDefaults().applyTo(xmin.getWidget());
        
        // Max value for x-axis
        GridDataFactory.fillDefaults().align(SWT.END, SWT.CENTER).applyTo(labelMax);
        GridDataFactory.fillDefaults().applyTo(xmax.getWidget());
        
        // Axis hide buttons
        GridDataFactory.fillDefaults().applyTo(axisHide);

        // Set the same width to both label rows
        composite.layout();
        GridDataFactory.fillDefaults().hint(xVariableLabel.getBounds().width, SWT.DEFAULT).align(SWT.END, SWT.CENTER).applyTo(nameLabel);
       
        Point size = composite.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        sc.setMinSize(size);
	}

}