/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.simantics.browsing.ui.swt.widgets.TrackedCombo;
import org.simantics.browsing.ui.swt.widgets.impl.ComboModifyListenerImpl;
import org.simantics.browsing.ui.swt.widgets.impl.ReadFactoryImpl;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupportImpl;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.RunnableWithObject;
import org.simantics.utils.datastructures.Triple;
import org.simantics.utils.ui.AdaptionUtils;

/**
 * Composite for range controls in chart series properties
 * @author Teemu Lempinen
 *
 */
public class RangeComposite extends Composite implements Widget {

    private Composite composite;

    public RangeComposite(Composite parent, ISessionContext context, WidgetSupport support, int style) {
        super(parent, style);
        support.register(this);
        GridLayoutFactory.fillDefaults().spacing(3, 0).margins(3, 0).applyTo(this);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(this);
        composite = this;
    }

    @Override
    public void setInput(final ISessionContext context, final Object input) {
        if(composite == null || composite.isDisposed())
            return;

        final Resource series = AdaptionUtils.adaptToSingle(input, Resource.class);

        RangeHandlerFactory f;
        try {
            f = context.getSession().syncRequest(new Read<RangeHandlerFactory>() {
                @Override
                public RangeHandlerFactory perform(ReadGraph graph)
                        throws DatabaseException {
                    return graph.adapt(series, RangeHandlerFactory.class);
                }
            });

        } catch (DatabaseException e) {
            //ExceptionUtils.logAndShowError("Insert something intelligent here.", e);
            return;
        }

        final RangeHandlerFactory factory = f;

        /*
         *  Listen to the enumerations assigned to the variable in this series.
         *  Listener is needed because the user can change the variableRVI for the series
         *  and that changes the options for range
         */
        context.getSession().asyncRequest(factory.getRequest(series), new Listener<LinkedHashMap<String, Resource>>() {

            @Override
            public void execute(LinkedHashMap<String, Resource> result) {
                if(isDisposed())
                    return;

                // Always modify the composite, even with null result
                composite.getDisplay().asyncExec(new RunnableWithObject(result) {
                    @Override
                    public void run() {
                        if(composite == null || composite.isDisposed())
                            return;

                        // Remove all content (even with null result)
                        for(Control child : composite.getChildren())
                            child.dispose();
                        
                        composite.layout();

                        if(getObject() == null) {
                        	// No range, print an em dash.
                        	Label label = new Label(composite, SWT.NONE);
                            label.setText("\u2014");
                            label.setEnabled(false);
                            GridDataFactory.fillDefaults().applyTo(label);
                            composite.layout();
                            return;
                        }

                        // New widgetSupport for the combos
                        WidgetSupportImpl support = new WidgetSupportImpl();

                        Label label;
                        TrackedCombo combo;
                        LinkedHashMap<?, ?> result = (LinkedHashMap<?, ?>)getObject();
                        Iterator<?> iterator = result.keySet().iterator();

                        // Set the width of the combo 
                        GridLayout gl = (GridLayout)composite.getLayout();
                        gl.numColumns = result.size();
                        
                        // For each array index (enumeration), create a label and a combo 
                        int index = 0;
                        while(iterator.hasNext()) {
                            Object key = iterator.next();
                            Composite c = new Composite(composite, SWT.NONE);
                            GridDataFactory.fillDefaults().applyTo(c);
                            GridLayoutFactory.fillDefaults().applyTo(c);

                            label = new Label(c, SWT.NONE);
                            label.setText((String)key);
                            GridDataFactory.fillDefaults().align(SWT.CENTER, SWT.END).applyTo(label);

                            combo = new TrackedCombo(c, support, SWT.READ_ONLY);
                            combo.setItemFactory(factory.getRangeItemFactory(index, (Resource)result.get(key)));
                            combo.setSelectionFactory(new RangeSelectionFactory(index));
                            combo.addModifyListener(new RangeModifyListener(index, result.size()));
                            GridDataFactory.fillDefaults().applyTo(combo.getWidget());
                            index++;
                        }

                        // Set input for the combos
                        support.fireInput(context, input);

                        /*
                         *  Find out if this composite is located in a scrolled composite. 
                         *  If it is, resize the scrolled composite
                         */
                        composite.layout();
                        Composite previousParent = composite.getParent();
                        for(int i = 0; i < 7 && previousParent != null; i++) {
                            previousParent.layout();
                            if(previousParent.getParent() instanceof ScrolledComposite) {
                                ScrolledComposite sc = (ScrolledComposite) previousParent.getParent();
                                Point size = previousParent.computeSize(SWT.DEFAULT, SWT.DEFAULT);
                                sc.setMinSize(size);
                                break;
                            }
                            previousParent = previousParent.getParent();
                        }
                    }
                });                    
            }

            @Override
            public void exception(Throwable t) {
                t.printStackTrace();
            }

            @Override
            public boolean isDisposed() {
                return composite == null || composite.isDisposed();
            }
        });
    }

    /**
     * 
     * @author Teemu Lempinen
     *
     */
    private class RangeSelectionFactory extends ReadFactoryImpl<Resource, String> {

        int index;

        /**
         * 
         * @param index Index of the enumeration in the variable
         */
        public RangeSelectionFactory(int index) {
            this.index = index;
        }

        public Object getIdentity(Object inputContents) {
            return new Triple<Object, Integer, Class<?>>(inputContents, index, getClass());
        }

        @Override
        public String perform(ReadGraph graph, Resource series) throws DatabaseException {
            String[] filter = graph.getPossibleRelatedValue(series, getIndexRelation(graph), Bindings.STRING_ARRAY);


            /*
             * If no filter was found or the index is not applicable, return the first index
             */
            String result = null;
            if(filter == null)
                result = getFirstIndex(graph, series, index);
            else if(filter.length < index)
                result = getFirstIndex(graph, series, index);      
            else
                result = filter[index];

            return result;
        }

    }



    /**
     * RangeModifyListener for modifying a range filter in chart series 
     * @author Teemu Lempinen
     *
     */
    private class RangeModifyListener  extends ComboModifyListenerImpl<Resource> {

        private int index, size;

        /**
         * 
         * @param index Index of the modified range filter
         * @param size Size of the whole filter (for situations where there is no filter)
         */
        public RangeModifyListener(int index, int size) {
            this.index = index;
            this.size = size;
        }

        @Override
        public void applyText(WriteGraph graph, Resource series, String text) throws DatabaseException {
            Resource filterRelation = getIndexRelation(graph);
            String[] filter = graph.getPossibleRelatedValue(series, filterRelation, Bindings.STRING_ARRAY);

            // If there is no filter, create a default filter with all indexes "All"
            if(filter == null) {
                filter = new String[size];
                for(int i = 0; i < filter.length; i++) {
                    filter[i] = getFirstIndex(graph, series, i);
                }
            }

            // Modify the filter index
            filter[index] = text;
            graph.claimLiteral(series, filterRelation, filter, Bindings.STRING_ARRAY);
        }
    }

    protected Resource getIndexRelation(ReadGraph graph) {
        JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
        return jfree.variableFilter;
    }
    
    private static String getFirstIndex(ReadGraph graph, Resource series, int index) throws DatabaseException {
        RangeHandlerFactory f = graph.adapt(series, RangeHandlerFactory.class);
        LinkedHashMap<String, Resource> map = graph.syncRequest(f.getRequest(series));
        if(map == null)
            return null;
        else {
            Resource enumeration = null;
            Iterator<Resource> iterator = map.values().iterator();
            for(int i = 0; i <= index && iterator.hasNext(); i++) {
                enumeration = iterator.next();
            }
            if(enumeration != null) {
                Map<String, Object> indexmap = f.getRangeItemFactory(index, enumeration).perform(graph, null);
                if(indexmap != null)
                    return indexmap.values().iterator().next().toString();
            }
        }

        return null;
    }
}
