/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.properties;

import org.simantics.browsing.ui.swt.widgets.impl.ReadFactoryImpl;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Quad;

/**
 * PropertyFactory for finding a boolean property. Supports also finding the 
 * property from a first occurrence of resource ConsistsOf type HasProperty   
 * 
 * @author Teemu Lempinen
 *
 */
public class BooleanPropertyFactory extends ReadFactoryImpl<Resource, Boolean> {

    final private String propertyURI;
    final private String typeURI;
    final private Boolean inverse;
    final private Boolean defaultValue;
    
    /**
     * PropertyFactory for finding a boolean property with propertyURI
     * 
     * @param propertyURI URI for the boolean property
     */
    public BooleanPropertyFactory(String propertyURI) {
        this(null, propertyURI, false);
    }
    
    /**
     * PropertyFactory for finding a boolean property with propertyURI.
     * 
     * Supports inverting the result (e.g. if required information is IsHidden, but database contains IsVisible)
     * 
     * @param propertyURI URI for the boolean property
     * @param inverse Invert the result?
     */
    public BooleanPropertyFactory(String propertyURI, boolean inverse) {
        this(null, propertyURI, inverse);
    }

    /**
     * PropertyFactory for finding a boolean property with propertyURI.
     *  
     * Finds the property for first ObjectWithType(resource, L0.ConsistsOf, type)
     *  
     * Supports inverting the result (e.g. if required information is IsHidden, but database contains IsVisible)
     *  
     * @param typeURI URI for a resource (resource ConsistsOf type) (null allowed)
     * @param propertyURI URI for the boolean property
     * @param inverse Invert the result?
     */
    public BooleanPropertyFactory(String typeURI, String propertyURI, boolean inverse) {
        this(typeURI, propertyURI, inverse, false);
    }
    
    /**
     * PropertyFactory for finding a boolean property with propertyURI.
     *  
     * Finds the property for first ObjectWithType(resource, L0.ConsistsOf, type)
     *  
     * Supports inverting the result (e.g. if required information is IsHidden, but database contains IsVisible)
     * 
     * @param typeURI URI for a resource (resource ConsistsOf type) (null allowed -> not used)
     * @param propertyURI URI for the boolean property
     * @param inverse Invert the result?
     * @param defaultValue default value
     */
    public BooleanPropertyFactory(String typeURI, String propertyURI, boolean inverse, boolean defaultValue) {
        this.propertyURI = propertyURI;
        this.inverse = inverse;
        this.typeURI = typeURI;
        this.defaultValue = defaultValue;
    }

    @Override
    public Object getIdentity(Object inputContents) {
        return new Quad<Resource, String, Object, Boolean>((Resource) inputContents, propertyURI, getClass(), defaultValue);
    }

    @Override
    public Boolean perform(ReadGraph graph, Resource r) throws DatabaseException {
        if(typeURI == null) {
            // if no typeUri, use the default resource r
            return getValue(graph, r);
        } else {
            // typeURI was defined, find the property for the first occurence of ConsistsOf type
            Resource type =  graph.getResource(typeURI);
            for(Resource o : graph.syncRequest(new ObjectsWithType(r, Layer0.getInstance(graph).ConsistsOf, type))) {
                // Returns the value for the first occurrence
                return getValue(graph, o);
            }
        }
        // if nothing was found with typeURI
        return false;
    }
    
    /**
     * Return the value for a Boolean literal possibly inverted (or default if resource != Boolean literal) 
     * 
     * @param graph ReadGraph
     * @param resource Literal Boolean resource 
     * @return value of the parameter (or default or inverted)
     * @throws DatabaseException
     */
    private Boolean getValue(ReadGraph graph, Resource resource) throws DatabaseException {
        Boolean value = graph.getPossibleRelatedValue(resource, graph.getResource(propertyURI), Bindings.BOOLEAN);
        if(value != null) {
            return !inverse.equals(value);
        } else {
            return defaultValue;
        }
    }
}