/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.ge;

import java.util.Collections;
import java.util.List;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.DropActionFactory;
import org.simantics.jfreechart.chart.properties.xyline.XYLineAxisAndVariablesTab;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.ui.AdaptionUtils;

/**
 * Drop action for explorer in {@link XYLineAxisAndVariablesTab}. This action is used for dropping 
 * both series on axis or another sries
 * 
 * @author Teemu Lempinen
 *
 */
public class SeriesDropAction implements DropActionFactory {

    @Override
    public Runnable create(ReadGraph g, Object target, Object source, int operation) throws DatabaseException {
        // Make sure that both target and source are resources
        Resource t = AdaptionUtils.adaptToSingle(target, Resource.class);
        Resource s = AdaptionUtils.adaptToSingle(source, Resource.class);
        
        if(t == null || s == null)
            return null;
        
        // Make sure that source and target are of correct type
        JFreeChartResource jfree = JFreeChartResource.getInstance(g);
        if(!g.isInstanceOf(s, jfree.Series))
            return null;
        if(!g.isInstanceOf(t, jfree.Series) && !g.isInstanceOf(t, jfree.Axis))
            return null;
        
        return getRunnable(t, s);
    }

    /**
     * Get the runnable for doing the drop action 
     * 
     * @param t target resource
     * @param s source resource
     * @return Runnable
     */
    private Runnable getRunnable(final Resource t, final Resource s) {
        Runnable runnable = new Runnable() {

            @Override
            public void run() {
                Simantics.getSession().asyncRequest(new WriteRequest() {

                    @Override
                    public void perform(WriteGraph graph) throws DatabaseException {
                        if(t == null || s == null) return;
                        JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
                        Layer0 l0 = Layer0.getInstance(graph);
                        Resource target = t;
                        Resource source = s;
                        Resource droppedOnSeries = null;

                        // Dropped a series over a series -> get target dataset
                        if(graph.isInstanceOf(target, jfree.Series)) {
                            droppedOnSeries = target;
                            Resource dataset = graph.getPossibleObject(target, l0.PartOf);
                            if(dataset != null)
                                target = dataset;
                        }

                        // Dropped a series over an axis -> get target dataset
                        if(graph.isInstanceOf(target, jfree.Axis)) {
                            Resource dataset = graph.syncRequest(new PossibleObjectWithType(target, jfree.Dataset_mapToRangeAxis_Inverse, jfree.Dataset));
                            if(dataset != null)
                                target = dataset;
                        }

                        // Move series to a dataset
                        if(graph.isInstanceOf(target, jfree.Dataset)) {
                            // Remove from old dataset if it was different than the new one
                            Resource sourceDataset = graph.getPossibleObject(source, l0.PartOf);
                            if(sourceDataset != null && !sourceDataset.equals(target)) {
                                Resource sourceSeriesList = graph.getPossibleObject(sourceDataset, jfree.Dataset_seriesList);
                                if(sourceSeriesList != null)
                                    ListUtils.removeElement(graph, sourceSeriesList, source);
                            }
                            graph.deny(source, l0.PartOf);

                            // Add to new dataset
                            Resource targetSeriesList = graph.getPossibleObject(target, jfree.Dataset_seriesList);
                            if(targetSeriesList == null) {
                                targetSeriesList = ListUtils.create(graph, Collections.<Resource>emptyList());
                                graph.claim(target, jfree.Dataset_seriesList, targetSeriesList);
                            }


                            // Series was dropped on another series. Move the dropped series to that place and recreate the list
                            if(droppedOnSeries != null) {
                                List<Resource> list = ListUtils.toList(graph, targetSeriesList);
                                int targetIndex = list.indexOf(droppedOnSeries);
                                if(list.contains(source))
                                    list.remove(source);
                                list.add(targetIndex, source);
                                graph.deny(target, jfree.Dataset_seriesList);
                                targetSeriesList = ListUtils.create(graph, list);
                                graph.claim(target, jfree.Dataset_seriesList, targetSeriesList);
                            } else {
                                ListUtils.insertFront(graph, targetSeriesList, Collections.singleton(source));
                            }

                            graph.claim(target, l0.ConsistsOf, source);
                        }
                    }
                });
            }
        };
        return runnable;
    }

}
