/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart.element;

import java.awt.BasicStroke;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Path2D;
import java.awt.geom.QuadCurve2D;
import java.awt.geom.Rectangle2D;

import org.jfree.chart.JFreeChart;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.adapter.ElementFactory;
import org.simantics.diagram.adapter.SyncElementFactory;
import org.simantics.diagram.elements.ElementPropertySetter;
import org.simantics.diagram.synchronization.CompositeHintSynchronizer;
import org.simantics.diagram.synchronization.IHintSynchronizer;
import org.simantics.diagram.synchronization.SynchronizationHints;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.diagram.synchronization.graph.RemoveElement;
import org.simantics.diagram.synchronization.graph.TransformSynchronizer;
import org.simantics.diagram.ui.DiagramModelHints;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.LifeCycle;
import org.simantics.g2d.element.handler.impl.DefaultTransform;
import org.simantics.g2d.element.handler.impl.StaticObjectAdapter;
import org.simantics.g2d.element.handler.impl.StaticSymbolImageInitializer;
import org.simantics.g2d.element.handler.impl.StaticSymbolImpl;
import org.simantics.g2d.image.Image;
import org.simantics.g2d.image.impl.ShapeImage;
import org.simantics.jfreechart.chart.IJFreeChart;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintContext.KeyOf;

/**
 * Element factory for creating chart elements to diagrams
 * 
 * @author Teemu Lempinen
 *
 */
public class ChartElementFactory extends SyncElementFactory {

    private static final String         CLASS_ID            = "Chart";
    public  static final ElementFactory INSTANCE            = new ChartElementFactory();
    public  static final Image          STATIC_IMAGE        = new ShapeImage(getChartShape(), null, new BasicStroke(1f), true);
    public  static final double         SYMBOL_CHART_SIZE   = 10.0;
    public  static final Key            KEY_CHART_COMPONENT = new KeyOf(Resource.class, "CHART_COMPONENT");     
    public  static final Key            KEY_CHART           = new KeyOf(JFreeChart.class, "CHART");       
    
    static Shape getChartShape() {
        
        Path2D path = new Path2D.Double();
        // First create the axis for a chart symbol
        path.moveTo(-SYMBOL_CHART_SIZE, -SYMBOL_CHART_SIZE);
        path.lineTo(-SYMBOL_CHART_SIZE,  SYMBOL_CHART_SIZE);
        path.lineTo( SYMBOL_CHART_SIZE,  SYMBOL_CHART_SIZE);

        // Then a curve to the chart
        QuadCurve2D curve = new QuadCurve2D.Double(
                -SYMBOL_CHART_SIZE + 1,  SYMBOL_CHART_SIZE - 1,
                SYMBOL_CHART_SIZE - 5,  SYMBOL_CHART_SIZE - 5,
                SYMBOL_CHART_SIZE - 1, -SYMBOL_CHART_SIZE + 3);

        // Connect shapes
        path.append(curve, false);
        return path;
    }

    // Hint synchronizer for synchronizing transform and bounds
    private static final IHintSynchronizer HINT_SYNCHRONIZER = new CompositeHintSynchronizer(
            TransformSynchronizer.INSTANCE);

    public static ElementClass create(ReadGraph graph, Resource chart) throws DatabaseException {
        return ElementClass.compile(
                new ChartSceneGraph(),
                new Initializer(chart),
                new StaticObjectAdapter(JFreeChartResource.getInstance(graph).ChartElement),
                DefaultTransform.INSTANCE,
                StaticSymbolImageInitializer.INSTANCE,
                new StaticSymbolImpl(STATIC_IMAGE),
                new ChartBoundsOutline(new Rectangle2D.Double(-20, -20, 60, 40)),
                new ElementPropertySetter(ChartSceneGraph.KEY_SG_NODE)
                ).setId(CLASS_ID);
    }

    public ElementClass create(ReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource elementType) throws DatabaseException {
        return create(graph, null);
    }

    @Override
    public void load(ReadGraph g, ICanvasContext canvas, final IDiagram diagram, Resource element, final IElement e) throws DatabaseException {

        Resource chartResource;
        ElementPropertySetter ps = e.getElementClass().getSingleItem(ElementPropertySetter.class);
        ps.loadProperties(e, element, g);
        
        AffineTransform at = DiagramGraphUtil.getAffineTransform(g, element);
        // Hack for disabling all rotations in chart elements
        double x = at.getTranslateX();
        double y = at.getTranslateY();
        at.setToRotation(0);
        at.setToTranslation(x, y);
        ElementUtils.setTransform(e, at); // Set hint transform without rotations
        ps.overrideProperty(e, "Transform", at); // Set property Transform without rotations
        
        e.setHint(SynchronizationHints.HINT_SYNCHRONIZER, HINT_SYNCHRONIZER);
        

        Object o = e.getHint(KEY_CHART_COMPONENT);
        if(o == null || !(o instanceof Resource)) {
            chartResource = g.getPossibleObject(element, JFreeChartResource.getInstance(g).ChartElement_component);
        } else {
            chartResource = (Resource)o;
        }

        if(chartResource == null || !g.hasStatement(chartResource))  {
            // Remove element if there is no chart resource for it        
            g.asyncRequest(new WriteRequest() {
                
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    new RemoveElement(
                            (Resource)diagram.getHint(DiagramModelHints.KEY_DIAGRAM_RESOURCE),
                            (Resource)e.getHint(ElementHints.KEY_OBJECT))
                    .perform(graph);
                }
            });
            return;
        }

        IJFreeChart ichart = g.adapt(chartResource, IJFreeChart.class);

        if(ichart != null) {
            JFreeChart chart = ichart.getChart();
            e.setHint(KEY_CHART, chart);
        }

    }


    /**
     * Initializer for setting a chart component for element
     * @author Teemu Lempinen
     *
     */
    static class Initializer implements LifeCycle {
        private static final long serialVersionUID = -5822080013184271204L;
        Object component;

        Initializer(Object component) {
            this.component = component;
        }

        @Override
        public void onElementCreated(IElement e) {
            if(component != null) e.setHint(KEY_CHART_COMPONENT, component);
        }

        @Override
        public void onElementActivated(IDiagram d, IElement e) {}
        @Override
        public void onElementDeactivated(IDiagram d, IElement e) {}
        @Override
        public void onElementDestroyed(IElement e) {}
    };
}
