package org.simantics.jfreechart.chart;

import java.awt.Dimension;
import java.awt.geom.Rectangle2D;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.UUID;

import org.apache.batik.dom.GenericDOMImplementation;
import org.apache.batik.svggen.SVGGraphics2D;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleActiveExperiment;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.layer0.Layer0;
import org.simantics.layer0.utils.direct.GraphUtils;
import org.simantics.operation.Layer0X;
import org.simantics.sysdyn.JFreeChartResource;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;

/**
 * Utilities for handling charts
 * 
 * @author Teemu Lempinen
 * @author Tuomas Miettinen
 *
 */
public class ChartUtils {
    
	public static final String emptyVariableName = "<Write variable name>";

    /**
     * Creates a new range axis of type jfree.NumberAxis to a plot
     * 
     * @param graph WriteGraph
     * @param plot Plot resource
     * @return Created number axis, null if not successful
     * @throws DatabaseException
     */
    public static Resource createNumberRangeAxis(WriteGraph graph, Resource plot) throws DatabaseException {
        Resource axis = null;
        JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
        Layer0 l0 = Layer0.getInstance(graph);

        if(plot != null) {
            // Create range axis
            axis = GraphUtils.create2(graph, jfree.NumberAxis,
                    l0.HasName, "NumberAxis" + UUID.randomUUID().toString(),
                    l0.HasLabel, NameUtils.findFreshLabel(graph, "Y-axis", plot),
                    jfree.Plot_rangeAxis_Inverse, plot,
                    l0.PartOf, plot);
            
            // Add range axis to the plot's range axis list
            Resource axisList = graph.getPossibleObject(plot, jfree.Plot_rangeAxisList);
            ArrayList<Resource> list = new ArrayList<Resource>();
            list.add(axis);
            if(axisList == null) {
                axisList = ListUtils.create(graph, list);
                graph.claim(plot, jfree.Plot_rangeAxisList, axisList);
            } else {
                ListUtils.insertBack(graph, axisList, list);
            }
        }
        
        return axis;
        
    }
    
    /**
     * Create a XYDataset and map it to axis
     * @param graph WriteGraph
     * @param plot Plot resource containing the dataset
     * @param domainAxis Mapped domain axis for the dataset
     * @param rangeAxis Mapped range axis for the dataset
     * @return created dataset or null if not successful
     * @throws DatabaseException
     */
    public static Resource createXYDataset(WriteGraph graph, Resource plot, Resource domainAxis, Resource rangeAxis) throws DatabaseException {
        if(plot == null || domainAxis == null || rangeAxis == null)
            return null;
        
        JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
        Layer0 l0 = Layer0.getInstance(graph);
        
        
        // Create a dataset for the axis
        Resource dataset = GraphUtils.create2(graph, jfree.XYDataset,
                l0.HasName, "XYDataset" + UUID.randomUUID().toString(),
                jfree.Dataset_mapToDomainAxis, domainAxis,
                jfree.Dataset_mapToRangeAxis, rangeAxis,
                jfree.Dataset_renderer, GraphUtils.create2(graph, jfree.XYLineRenderer),
                l0.PartOf, plot);
        
        return dataset;
    }

    /**
     * Creates a new series to a dataset
     * @param graph WriteGraph
     * @param dataset Dataset for the new series
     * @return created series or null if unsuccessful
     * @throws DatabaseException
     */
    public static Resource createSeries(WriteGraph graph, Resource dataset, String rvi) throws DatabaseException {
        if(dataset == null) return null;
        
        JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
        Layer0 l0 = Layer0.getInstance(graph);
        // Create series
        Resource series = GraphUtils.create2(graph, jfree.Series,
                l0.HasName, "Series" + UUID.randomUUID().toString(),
                jfree.variableRVI, rvi == null ? " " + emptyVariableName : rvi,
                l0.PartOf, dataset);

        // Add series to the dataset's series list
        Resource seriesList = graph.getPossibleObject(dataset, jfree.Dataset_seriesList);
        ArrayList<Resource> list = new ArrayList<Resource>();
        list.add(series);
        if(seriesList == null) {
            seriesList = ListUtils.create(graph, list);
            graph.claim(dataset, jfree.Dataset_seriesList, seriesList);
        } else {
            ListUtils.insertBack(graph, seriesList, list);
        }
        
        return series;
    }

    /**
     * Find the current realization uri
     * 
     * @param graph ReadGraph
     * @param chartComponent A resource from a chart (consistsOf relation in a chart)
     * @return current realization uri
     * @throws DatabaseException
     */
    public static String getCurrentRealizationURI(ReadGraph graph, Resource chartComponent) throws DatabaseException {
        // Find the model where the chart is located
        Resource model = graph.syncRequest(new PossibleModel(chartComponent)); 
        if(model == null)
            return null;
        
        // Find the variable realization of the current experiment
        String realizationURI = null;
        Resource realization = graph.syncRequest(new PossibleActiveExperiment(model));
        if (realization == null) {
            Layer0X L0X = Layer0X.getInstance(graph);
            realization = graph.getPossibleObject(model, L0X.HasBaseRealization);
        }
        if (realization != null)
            realizationURI = graph.getURI(realization);
        
        return realizationURI;        
    }
    
    public static void writeSVG(org.jfree.chart.JFreeChart chart, Rectangle2D bounds, Writer out) throws IOException {
        // Get a DOMImplementation.
        DOMImplementation domImpl = GenericDOMImplementation.getDOMImplementation();

        // Create an instance of org.w3c.dom.Document.
        Document document = domImpl.createDocument(null, "svg", null);

        // Create an instance of the SVG Generator.
        SVGGraphics2D svgGenerator = new SVGGraphics2D(document);

        // Paint chart panel
        svgGenerator.setSVGCanvasSize(new Dimension((int)bounds.getWidth(), (int)bounds.getHeight()));
        chart.draw(svgGenerator, bounds);

        // Finally, write to out
        svgGenerator.stream(out, false);

    }

    public static void writeSVG(org.jfree.chart.JFreeChart chart, Rectangle2D bounds, File file) throws IOException {
        OutputStream outputStream = new FileOutputStream(file);
        Writer out = new OutputStreamWriter(outputStream, "UTF-8");
        writeSVG(chart, bounds, out);
        outputStream.flush();
        outputStream.close();
    }
}
