/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart.chart;

import java.awt.Frame;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.awt.SWT_AWT;
import org.eclipse.swt.widgets.Composite;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.simantics.Simantics;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncListener;
import org.simantics.db.request.Read;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.utils.RunnableWithObject;

/**
 * Composite containing a single chart defined by a JFreeChart.Chart
 * 
 * @author Teemu Lempinen
 *
 */
public class ChartComposite extends Composite {

    private Frame frame;
    private ChartPanel panel;
    private Composite composite;
    private IJFreeChart chart;

    /**
     * A new ChartComposite with a definition in chartResourceURI
     * @param parent Composite
     * @param chartResourceURI URI for a JFreeChart.Chart definition
     * @param style SWT style
     */
    public ChartComposite(Composite parent, final String chartResourceURI, int style) {
        super(parent, style | SWT.NO_BACKGROUND | SWT.EMBEDDED);

        try {
            Resource chartResource = Simantics.getSession().syncRequest(
                    (Read<Resource>) graph -> graph.getPossibleResource(chartResourceURI));
            if(chartResource != null)
                CreateContent(chartResource);
        } catch (DatabaseException e) {
            e.printStackTrace();
        }

        this.addDisposeListener(e -> doDispose());
    }

    /**
     * A new ChartComposite with a chartResource definition
     * @param parent Composite
     * @param chartResource JFreeChart.Chart resource
     * @param style SWT style
     */
    public ChartComposite(Composite parent, final Resource chartResource, int style) {
        super(parent, style | SWT.NO_BACKGROUND | SWT.EMBEDDED);
        CreateContent(chartResource);
    }

    public void doDispose() {
        if (chart != null) {
            chart.dispose();
            chart = null;
        }
    }

    /**
     * This query does not implement equals or hashCode by purpose. The result is not reusable.
     * 
     * @author luukkainen
     */
    protected static class ChartRead implements Read<IJFreeChart> {
        private Resource chartResource;

        public ChartRead(Resource chartResource) {
            this.chartResource = chartResource;
        }

        @Override
        public IJFreeChart perform(ReadGraph graph) throws DatabaseException {
            // Adapt chartResource to a chart (XY, pie, bar, ...)
            if(graph.isInstanceOf(chartResource, JFreeChartResource.getInstance(graph).Chart)) {
                return  graph.adaptUnique(chartResource, IJFreeChart.class);
            } else {
                return null;
            }
        }
    }

    /**
     * Creates and displays the chart defined in chartResource
     * @param chartResource
     */
    private void CreateContent(final Resource chartResource) {
        composite = this;

        GridLayoutFactory.fillDefaults().applyTo(composite);
        GridDataFactory.fillDefaults().grab(true, true).applyTo(composite);
        frame = SWT_AWT.new_Frame(composite);

        // Add a listener displaying the contents of the chart. Chart is re-drawn if the definition changes
        Simantics.getSession().asyncRequest(new ChartRead(chartResource), new AsyncListener<IJFreeChart>() {

            @Override
            public boolean isDisposed() {
                return composite.isDisposed();
            }

            @Override
            public void execute(AsyncReadGraph graph, IJFreeChart chart) {
                if(chart == null || composite.isDisposed())
                    return;
                if (ChartComposite.this.chart != null)
                    ChartComposite.this.chart.dispose();
                ChartComposite.this.chart = chart;
                JFreeChart jfreeChart = chart.getChart();
                // Display the result chart
                if (composite.isDisposed())
                    return;
                composite.getDisplay().asyncExec(new RunnableWithObject(jfreeChart) {

                    @Override
                    public void run() {
                        if(composite.isDisposed())
                            return;
                        if(panel != null)
                            frame.remove(panel);
                        composite.layout();
                        JFreeChart chart = (JFreeChart)getObject();
                        //panel = new ChartPanel(chart, false, true, true, true, true);
                        panel = new ChartPanel(chart,
                                               ChartPanel.DEFAULT_WIDTH,
                                               ChartPanel.DEFAULT_HEIGHT,
                                               ChartPanel.DEFAULT_MINIMUM_DRAW_WIDTH,
                                               ChartPanel.DEFAULT_MINIMUM_DRAW_HEIGHT,
                                               ChartPanel.DEFAULT_MAXIMUM_DRAW_WIDTH, 
                                               ChartPanel.DEFAULT_MAXIMUM_DRAW_HEIGHT,
                                               false,
                                               false, true, true, true, true);
                        frame.add(panel);
//                        frame.repaint();
                        frame.validate();
                    }
                });
            }

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                throwable.printStackTrace();
            }
        });
    }

}
