/*******************************************************************************
 * Copyright (c) 2007, 2025 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.jfreechart;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSource;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.jfreechart.internal.Activator;
import org.simantics.layer0.Layer0;
import org.simantics.sysdyn.JFreeChartResource;
import org.simantics.ui.dnd.LocalObjectTransfer;
import org.simantics.utils.datastructures.Pair;

/**
 * Header of a chart element in {@link ChartPanel}. Only this header is
 * shown if a chart is minimized. If a chart is expanded, this header is added
 * to the charts {@link ChartPanelElement}. 
 * 
 * @author Teemu Lempinen
 *
 */
public class ChartPanelHeader extends Composite {

    public static int HEADER_MINIMUM_WIDTH = 250;
    private ChartPanelElement element;
    private Resource resource;
    private LocalResourceManager rm;
    private Label name;
    private Canvas iconCanvas;
    private Image icon;
    private ToolItem minimize, remove;
    private Color defaultColor, darker, evenDarker;
    private Image gradientBackgroundImage, borderImage;

    private static ImageDescriptor closeDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/close.gif"));
    private static ImageDescriptor minimizeDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/minimize.gif"));
    private static ImageDescriptor maximizeDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/maximize.gif"));
    private static ImageDescriptor lineChartDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/chart_line_light.png"));
    private static ImageDescriptor barChartDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/chart_bar_light.png"));
    private static ImageDescriptor pieChartDescriptor = ImageDescriptor.createFromURL(Activator.getDefault().getBundle().getResource("icons/chart_pie_light.png"));

    private Image closeImage;
    private Image minimizeImage;
    private Image maximizeImage;
    private Image lineChartImage;
    private Image barChartImage;
    private Image pieChartImage;

    /**
     * Chart panel header with minimize and close buttons.
     * 
     * @param parent The composite where the header is added
     * @param panel The {@link ChartPanel} containing the header
     * @param name The name of the chart
     * @param style he Style of the created chart element
     */
    public ChartPanelHeader(Composite c, ChartPanelElement element, Resource chartResource, int style) {
        super(c, style);
        this.resource = chartResource;
        this.element = element;
        this.rm = new LocalResourceManager(JFaceResources.getResources(), c);

        createColors(rm);
        createImages(rm);

        GridLayoutFactory.fillDefaults().margins(3, 0).numColumns(3).applyTo(this);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(this);

        // Colors

        // Chart icon
        iconCanvas = new Canvas (this, SWT.NONE);
        GridDataFactory.fillDefaults().align(SWT.CENTER, SWT.CENTER).hint(16, 16).applyTo(iconCanvas);
        iconCanvas.addPaintListener(e -> {
            if(icon != null)
                e.gc.drawImage (icon, 0, 0);
        });

        // Label for the chart name (also minimize/expand)
        name = new Label(this, SWT.NONE);

        try {
            // name updater
            Pair<String, Image> result = Simantics.getSession().syncRequest(new Read<Pair<String, Image>>() {

                @Override
                public Pair<String, Image> perform(ReadGraph graph) throws DatabaseException {
                    JFreeChartResource jfree = JFreeChartResource.getInstance(graph);
                    Layer0 l0 = Layer0.getInstance(graph);
                    String label = NameUtils.getSafeLabel(graph, resource);
                    Image image = null;
                    Resource plot = graph.syncRequest(new PossibleObjectWithType(resource, l0.ConsistsOf, jfree.Plot));
                    if(plot != null) {
                        if(graph.isInstanceOf(plot, jfree.CategoryPlot))
                            image = barChartImage;
                        else if(graph.isInstanceOf(plot, jfree.PiePlot))
                            image = pieChartImage;
                        else
                            image = lineChartImage;
                    }
                    return new Pair<String, Image>(label, image);
                }

            }, new Listener<Pair<String, Image>>() {

                @Override
                public void execute(final Pair<String, Image> result) {
                    if(result == null)
                        return;

                    name.getDisplay().asyncExec(new Runnable() {

                        @Override
                        public void run() {
                            if(!name.isDisposed() && result.first != null)
                                name.setText(result.first);

                            if(!iconCanvas.isDisposed() && result.second != null) {
                                icon = result.second;
                                iconCanvas.redraw();
                                ChartPanelHeader.this.layout();
                            }
                        }
                    });
                }

                @Override
                public void exception(Throwable t) {
                    t.printStackTrace();
                }

                @Override
                public boolean isDisposed() {
                    return name.isDisposed();
                }

            });
            name.setText(result.first);
        } catch (DatabaseException e) {
            e.printStackTrace();
            name.setText("No label");
        }
        GridDataFactory.fillDefaults().grab(true, false).applyTo(name);

        ToolBar toolbar = new ToolBar(this, SWT.FLAT);
        // item for minimizing/expanding chart
        minimize = new ToolItem(toolbar, SWT.PUSH);
        minimize.addSelectionListener(new MinimizeListener());
        if(isMinimized()) {
            minimize.setToolTipText("Expand");
            minimize.setImage(maximizeImage);
        } else {
            minimize.setToolTipText("Minimize");
            minimize.setImage(minimizeImage);
        }

        // item for closing/removing the chart
        remove = new ToolItem(toolbar, SWT.PUSH);
        remove.setImage(closeImage);
        remove.addSelectionListener(new RemoveChartListener());
        remove.setToolTipText("Remove");


        /* ********************************
         * DnD 
         * ********************************/

        // Allow data to be copied or moved from the drag source
        int operations = DND.DROP_MOVE;
        source = new DragSource(name, operations);

        // Provide data in Text format
        Transfer[] types = new Transfer[] {  LocalObjectTransfer.getTransfer() };
        source.setTransfer(types);
        dragSourceListener = new DragSourceListener() {

            @Override
            public void dragStart(DragSourceEvent event) {
                if(name.isDisposed())
                    event.doit = false;
                event.detail = DND.DROP_LINK;

            }

            @Override
            public void dragSetData(DragSourceEvent event) {
                event.data = new StructuredSelection(resource);
            }

            @Override
            public void dragFinished(DragSourceEvent event) {
            }
        };  
        source.addDragListener(dragSourceListener);

        name.addDisposeListener(new DisposeListener() {

            @Override
            public void widgetDisposed(DisposeEvent e) {
                if(dragSourceListener != null && source != null && !source.isDisposed()) {
                    source.removeDragListener(dragSourceListener);
                }
            }
        });

        gradientBackgroundImage = getGradientBackgroundImage();
        this.setBackgroundImage(gradientBackgroundImage);
        this.setBackgroundMode(SWT.INHERIT_FORCE);

        this.addListener(SWT.MouseEnter, new EnterListener());
        this.addListener(SWT.MouseExit, new ExitListener());

        for(Control child : this.getChildren()) {
            child.addListener(SWT.MouseEnter, new EnterListener());
            child.addListener(SWT.MouseExit, new ExitListener());
        }

        addDisposeListener(e -> {
            if (gradientBackgroundImage != null)
                gradientBackgroundImage.dispose();
            if (borderImage != null)
                borderImage.dispose();
        });
    }

    private void createImages(ResourceManager rm) {
        closeImage     = (Image) rm.get(closeDescriptor);
        minimizeImage  = (Image) rm.get(minimizeDescriptor);
        maximizeImage  = (Image) rm.get(maximizeDescriptor);
        lineChartImage = (Image) rm.get(lineChartDescriptor);
        barChartImage  = (Image) rm.get(barChartDescriptor);
        pieChartImage  = (Image) rm.get(pieChartDescriptor);
    }

    private class EnterListener implements org.eclipse.swt.widgets.Listener {
        public void handleEvent(Event event) {
            var oldImg = borderImage;
            borderImage = getHighlightedGradientBackgroundImage();
            ChartPanelHeader.this.setBackgroundImage(borderImage);
            if (oldImg != null)
                oldImg.dispose();
        }
    }

    private class ExitListener implements org.eclipse.swt.widgets.Listener {
        public void handleEvent(Event event) {
            var oldImg = gradientBackgroundImage;
            gradientBackgroundImage = getGradientBackgroundImage();
            ChartPanelHeader.this.setBackgroundImage(gradientBackgroundImage);
            if (oldImg != null)
                oldImg.dispose();
        }
    }

    private void createColors(ResourceManager rm) {
        var d = getDisplay();
        defaultColor = d.getSystemColor(SWT.COLOR_WIDGET_BACKGROUND);
        int r = defaultColor.getRed();
        int g = defaultColor.getGreen();
        int b = defaultColor.getBlue();

        try {
            darker = new Color(d, r - 30, g - 30, b - 30);
        } catch (IllegalArgumentException e) {
            // Do nothing, use the default color
            darker = defaultColor;
        }

        try {
            evenDarker = new Color(d, r - 50, g - 50, b - 50);
        } catch (IllegalArgumentException e) {
            // Do nothing, use the default color
            evenDarker = defaultColor;
        }
    }

    private Image createVerticalGradientImage(Color fg, Color bg, int h) {
        var img = new Image(this.getDisplay(), 1, h);
        var gc = new GC(img);
        gc.setForeground(fg);
        gc.setBackground(bg);
        gc.fillGradientRectangle(0, 0, 1, h, true);
        gc.dispose();
        return img;
    }

    private Image getHighlightedGradientBackgroundImage() {
        this.layout();
        Point size = this.getSize();
        return createVerticalGradientImage(defaultColor, evenDarker, Math.max(1, size.y));
    }

    private Image getGradientBackgroundImage() {
        this.layout();
        Point size = this.computeSize(SWT.DEFAULT, SWT.DEFAULT);
        return createVerticalGradientImage(defaultColor, darker, Math.max(1, size.y));
    }

    private DragSourceListener dragSourceListener;
    private DragSource source;

    /**
     * Return true if this element is minimized, false if expanded
     * @return true if this element is minimized, false if expanded
     */
    private boolean isMinimized() {
        return element.isMinimized();
    }

    /**
     * Listener to minimize chart button. Expands and minimizes 
     * the chart of this header.
     * 
     * @author Teemu Lempinen
     *
     */
    private class MinimizeListener implements SelectionListener {
        @Override
        public void widgetSelected(SelectionEvent e) {
            if(ChartPanelHeader.this.isDisposed())
                return;

            element.toggleMinimize(true);

            if(!name.isDisposed() && !minimize.isDisposed()) {
                if(isMinimized()) {
                    minimize.setToolTipText("Expand");
                } else {
                    minimize.setToolTipText("Minimize");
                }
            }
        }

        @Override
        public void widgetDefaultSelected(SelectionEvent e) {
            widgetSelected(e);
        }

    }

    /**
     * Listener for removing this chart from the chart panel.
     * 
     * @author Teemu Lempinen
     *
     */
    private class RemoveChartListener implements SelectionListener {
        @Override
        public void widgetSelected(SelectionEvent e) {
            if(!ChartPanelHeader.this.isDisposed()) {
                element.remove();
            }
        }

        @Override
        public void widgetDefaultSelected(SelectionEvent e) {
            widgetSelected(e);
        }

    }
}
