package org.simantics.graph.refactoring;

import java.util.Arrays;

import org.simantics.databoard.util.URIStringUtils;
import org.simantics.graph.query.Path;
import org.simantics.graph.query.PathChild;
import org.simantics.graph.query.TransferableGraphConversion;
import org.simantics.graph.query.UriUtils;
import org.simantics.graph.refactoring.MappingSpecification.MappingRule;
import org.simantics.graph.representation.External;
import org.simantics.graph.representation.Identity;
import org.simantics.graph.representation.Internal;
import org.simantics.graph.representation.Root;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.graph.representation.TransferableGraphUtils;
import org.simantics.graph.store.GraphStore;
import org.simantics.graph.store.IdentityStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.set.hash.TIntHashSet;

public class GraphRefactoringUtils {
	private static final Logger LOGGER = LoggerFactory.getLogger(GraphRefactoringUtils.class);
	/**
	 * Moves an external resource. Returns true if did something.
	 * @param parentsAffected 
	 */
	public static boolean moveExternal(TransferableGraph1 tg, IdentityStore ids, Path from, PathChild to, TIntHashSet parentsAffected) throws GraphRefactoringException {
		// Find from id
		int fromId = ids.pathToId(from);
		if(fromId < 0)
			return false;
		if(ids.isNewResource(fromId))
			throw new GraphRefactoringException("Cannot move internal resource " + from + ".");

		// Remove old identity
		int parentId = ids.removeIdentity(fromId);
		if(parentId >= 0)
			parentsAffected.add(parentId);

		// Find parent id
		int toParentId = ids.createPathToId(to.parent);
		if(ids.hasChild(toParentId, to.name)) {
			LOGGER.info("refactor statements from " + from + " to " + to);
			//throw new GraphRefactoringException("External reference to " + to + " already exists.");
			int toId = ids.pathToId(to);
			int[] statements = tg.statements;
			for(int i=0;i<tg.statements.length;++i) {
				if(statements[i] == fromId) statements[i] = toId;
			}
			return true;
		}

		// Set new identity
		ids.setIdentity(fromId, toParentId, to.name);
		return true;
	}

	public static void refactor(TransferableGraph1 tg, IdentityStore ids, MappingSpecification spec, TIntHashSet parentsAffected) throws GraphRefactoringException {
		for(MappingRule rule : spec.getRules()) {
			if(!(rule.to instanceof PathChild))
				throw new GraphRefactoringException("Invalid target URI " + rule.to);
			if(!moveExternal(tg, ids, rule.from, (PathChild)rule.to, parentsAffected))
				LOGGER.warn("Didn't find " + rule.from);
		}
	}

	public static boolean fixIncorrectRoot(Identity[] ids) {
		for(int i=0;i<ids.length;++i) {
			Identity id = ids[i];
			if(id.definition instanceof External) {
				External ext = (External)id.definition;
				if(ext.parent == -1 && (ext.name.equals("http:/") || ext.name.equals(""))) {
					id.definition = new Root("", "");
					return true;
				}
			}
		}
		return false;
	}

	public static void fixOntologyExport(TransferableGraph1 tg) {

		fixIncorrectRoot(tg.identities);
		fixOntologyRoot(tg, true);

	}

	private static Identity recursePath(TransferableGraph1 tg, String path) {

		Identity extId = TransferableGraphUtils.findExternal(tg, path);
		if(extId != null) return extId;
		if("http://".equals(path)) return TransferableGraphUtils.findRootWithName(tg, "");
		String[] parts = URIStringUtils.splitURI(path);
		Identity parentId = recursePath(tg, parts[0]);
		tg.identities = Arrays.copyOf(tg.identities, tg.identities.length+1);
		Identity childIdentity = new Identity(tg.resourceCount++, new External(parentId.resource, URIStringUtils.unescape(parts[1])));
		tg.identities[tg.identities.length-1] = childIdentity;
		return childIdentity;

	}

	public static void fixOntologyRoot(TransferableGraph1 tg, boolean tryToFix) {

		Identity[] ids = tg.identities;
		for(int i=0;i<ids.length;++i) {
			Identity id = ids[i];
			if(id.definition instanceof Root) {
				Root ext = (Root)id.definition;
				if(ext.name.startsWith("http://")) {

					String[] parts = URIStringUtils.splitURI(ext.name);
					Identity path = recursePath(tg, parts[0]);
					id.definition = new Internal(path.resource, URIStringUtils.unescape(parts[1]));

					GraphStore store = TransferableGraphConversion.convert(tg);
					int rootId = store.identities.createPathToId(UriUtils.uriToPathUnescaped(ext.name));
					propagateNewMarks(store.identities, rootId);

					TransferableGraph1 tgNew = TransferableGraphConversion.convert(store);

					tg.resourceCount = tgNew.resourceCount;
					tg.identities = tgNew.identities;
					tg.values = tgNew.values;
					tg.statements = tgNew.statements;

					return;

				} else if (ext.type.startsWith("http://")) {
					String first = "http://Projects/Development%20Project";
					Identity path = recursePath(tg, first);
					id.definition = new Internal(path.resource, ext.name);

					GraphStore store = TransferableGraphConversion.convert(tg);
					int rootId = store.identities.createPathToId(UriUtils.uriToPathUnescaped(first + "/" + ext.name));
					propagateNewMarks(store.identities, rootId);

					TransferableGraph1 tgNew = TransferableGraphConversion.convert(store);

					tg.resourceCount = tgNew.resourceCount;
					tg.identities = tgNew.identities;
					tg.values = tgNew.values;
					tg.statements = tgNew.statements;

				}
			}
		}

	}

	public static int relocateRelativeRoot(TransferableGraph1 tg, String uri) {

		Identity[] ids = tg.identities;
		for(int i=0;i<ids.length;++i) {
			Identity id = ids[i];
			if(id.definition instanceof Root) {
				Root ext = (Root)id.definition;
				if (ext.type.startsWith("http://")) {

					int originalResourceCount = tg.resourceCount;

					String[] parts = URIStringUtils.splitURI(uri);

					Identity path = recursePath(tg, parts[0]);
					id.definition = new Internal(path.resource, parts[1]);

					GraphStore store = TransferableGraphConversion.convert(tg);
					int rootId = store.identities.createPathToId(UriUtils.uriToPathUnescaped(uri));
					propagateNewMarks(store.identities, rootId);

					TransferableGraph1 tgNew = TransferableGraphConversion.convert(store);

					int newResources = tgNew.resourceCount - originalResourceCount;

					tg.resourceCount = tgNew.resourceCount;
					tg.identities = tgNew.identities;
					tg.values = tgNew.values;
					tg.statements = tgNew.statements;

					return newResources;

				}
			}
		}

		return 0;

	}

	private static void propagateNewMarks(IdentityStore identities, int resource) {
		if(identities.markNew(resource)) {
			for(int child : identities.getChildren(resource))
				propagateNewMarks(identities, child);
		}
	}

	public static void unfixIncorrectRoot(Identity[] ids) {
		for(int i=0;i<ids.length;++i) {
			Identity id = ids[i];
			if(id.definition instanceof Root) {
				Root root = (Root)id.definition;
				if(root.name.equals("") && root.type.equals("")) {
					id.definition = new External(-1, "http:/");
					return;
				}
			}
		}
	}

}
