/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.graph.db;

import java.util.HashSet;

import org.simantics.db.Session;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.common.request.WriteOnlyRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.InitSupport;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.scl.reflection.OntologyVersions;

public class CoreInitialization {

	static final public String LAYER0 = OntologyVersions.getInstance().currentVersion("http://www.simantics.org/Layer0-0.0/");
	static final HashSet<InitSupport.Builtin> BUILTINS =
		new HashSet<InitSupport.Builtin>();
	static public HashSet<InitSupport.Builtin> getBuiltins() {
	    return BUILTINS;
	}
	private static int builtinId = 0;
	private static void addBuiltin(String uri) {
		BUILTINS.add(new InitSupport.Builtin(uri, ++builtinId));
	}
	private static void addLayer0Builtin(String name) {
		addBuiltin(LAYER0 + name);
	}

	static {
		// IMPORTANT NOTICE:
		// DO NOT alter the order of these definitions in any way
		// unless you deliberately want to make the system incompatible
		// with databases that have been created before changing these.
		addLayer0Builtin("InstanceOf");
		addLayer0Builtin("Inherits");
		addLayer0Builtin("SubrelationOf");
		addLayer0Builtin("InverseOf");

		addLayer0Builtin("HasName");
		addLayer0Builtin("NameOf");

		addLayer0Builtin("ConsistsOf");
		addLayer0Builtin("PartOf");

		addLayer0Builtin("String");
		addLayer0Builtin("Library");
		addLayer0Builtin("FunctionalRelation");
		addLayer0Builtin("SuperrelationOf");
		addLayer0Builtin("Asserts");
		addLayer0Builtin("HasInstance");
		addLayer0Builtin("HasPredicate");
		addLayer0Builtin("HasPredicateInverse");
		addLayer0Builtin("HasObject");

		// #7016: This bogus URI replaces the builtin entry
		// that was originally http://Projects.
		//
		// addBuiltin("http://Projects") was removed because its existence
		// resulted in a non-functioning URI space because:
		//
		// 1. http://Projects was added as a "built-in resource" when the database
		//    is first initialized which means that ReadGraph.getResource("http://Projects")
		//    would succeed even when the resource was not yet properly imported
		//    from the project ontology.
		//
		// 2. When the project ontology is imported, a new resource is
		//    created to represent http://Projects instead of using the existing
		//    uninitialized "built-in resource". The L0.ExternalEntity changes made
		//    to the standard transferable graph import in this patch caused the
		//    code to find the built-in resource for "http://Projects" and think
		//    that it has to be a resource that's already properly initialized.
		//    This led to the TG import not initializing the resource at all.
		//    Previously there was a bug here as well but it was hidden by the fact
		//    that the system would import the actual/new "http://Projects"
		//    resource from the ontology anyway. This effectively left the
		//    uninitialized built-in resource just hanging in the database as
		//    trash.
		//addBuiltin("http://Projects");
		addBuiltin("75e23fb39121b4b37cf2e6a26ccb34c52f77522d-do-not-remove");
		addBuiltin("http:/");
		addLayer0Builtin("ExternalEntity");
		addLayer0Builtin("defaultMutableClusterSet");
		addLayer0Builtin("defaultImmutableClusterSet");
	}

    public static void initializeBuiltins(Session session) throws DatabaseException {
        session.getService(InitSupport.class).addBuiltins(BUILTINS);
    }

	public static long[] initializeGraph(final Session session, TransferableGraph1 tg) throws DatabaseException {
		final TransferableGraphImportProcess process = new TransferableGraphImportProcess(tg,
				new ImportAdvisor());
		session.syncRequest(new WriteOnlyRequest() {
			@Override
			public void perform(WriteOnlyGraph graph) throws DatabaseException {
			    graph.markUndoPoint();

//				// Hackety hack
//				session.getService(XSupport.class).setImmutable(session.getRootLibrary(), true);

				process.initialPrepare(graph);
				process.write(graph);
				graph.clearUndoList(this);

			}
		});
		return process.getResourceIds(
				session.getService(SerialisationSupport.class)
			);
	}

}
