package org.simantics.diagram.synchronization.graph;

import gnu.trove.list.array.TIntArrayList;
import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.RelativeReference;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.diagram.connection.RouteGraph;
import org.simantics.diagram.connection.RouteLine;
import org.simantics.diagram.connection.RouteLink;
import org.simantics.diagram.connection.RouteNode;
import org.simantics.diagram.connection.RoutePoint;
import org.simantics.diagram.connection.RouteTerminal;
import org.simantics.diagram.flag.RouteGraphConnectionSplitter;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.structural.stubs.StructuralResource2;

/**
 * Encodes modifications to a route graph.
 * @author Hannu Niemist&ouml;
 */
public class RouteGraphModification {
    // Identification information
    int lineCount;
    int terminalCount;
    int[] links;

    // Modification information
    ArrayList<Modi> modis = new ArrayList<Modi>(); 

    // Two different ways to identify
    Resource[] resources;    
    String[] terminalIdentifiers;

    // Auxiliary
    TObjectIntHashMap<RouteNode> idMap;
    Resource connection;

    public interface Modi {
    }

    public static class UpdateLine implements Modi {
        int id;
        double position;
        boolean isHorizontal;

        public UpdateLine(int id, double position, boolean isHorizontal) {
            this.id = id;
            this.position = position;
            this.isHorizontal = isHorizontal;
        }

        public UpdateLine(String text) {
            String[] parts = text.split("\\$");
            this.id = Integer.parseInt(parts[0]);
            this.position = Double.parseDouble(parts[1]);
            this.isHorizontal = Boolean.parseBoolean(parts[2]);
        }

        @Override
        public String toString() {
            return "M" + id + "$" + position + "$" + isHorizontal;
        }
    }

    public static class RemoveLink implements Modi {
        int a;
        int b;

        public RemoveLink(int a, int b) {
            this.a = a;
            this.b = b;
        }

        public RemoveLink(String text) {
            String[] parts = text.split("\\$");
            this.a = Integer.parseInt(parts[0]);
            this.b = Integer.parseInt(parts[1]);
        }

        @Override
        public String toString() {
            return "r" + a + "$" + b;
        }
    }

    public static class RemoveLine implements Modi {
        int a;

        public RemoveLine(int a) {
            this.a = a;
        }

        public RemoveLine(String text) {
            this.a = Integer.parseInt(text);
        }

        @Override
        public String toString() {
            return "R" + a;
        }
    }

    public static class CreateLink implements Modi {
        int a;
        int b;

        public CreateLink(int a, int b) {
            this.a = a;
            this.b = b;
        }

        public CreateLink(String text) {
            String[] parts = text.split("\\$");
            this.a = Integer.parseInt(parts[0]);
            this.b = Integer.parseInt(parts[1]);
        }

        @Override
        public String toString() {
            return "c" + a + "$" + b;
        }
    }

    public static class CreateLine implements Modi {
        double position;
        boolean isHorizontal;

        public CreateLine(double position, boolean isHorizontal) {
            this.position = position;
            this.isHorizontal = isHorizontal;
        }

        public CreateLine(String text) {
            String[] parts = text.split("\\$");
            this.position = Double.parseDouble(parts[0]);
            this.isHorizontal = Boolean.parseBoolean(parts[1]);
        }

        @Override
        public String toString() {
            return "C" + position + "$" + isHorizontal;
        }
    }

    public static class Split implements Modi {
        int[] interface1;
        int[] interface2; 
        int[] lines2; 
        int[] terminals1; 
        int[] terminals2;
        boolean isHorizontal;
        boolean invertFlagRotation;
        double isectX;
        double isectY;
        
        public Split(int[] interface1, int[] interface2, int[] lines2,
                int[] terminals1, int[] terminals2, boolean isHorizontal,
                boolean invertFlagRotation,
                double isectX, double isectY) {
            this.interface1 = interface1;
            this.interface2 = interface2;
            this.lines2 = lines2;
            this.terminals1 = terminals1;
            this.terminals2 = terminals2;
            this.isHorizontal = isHorizontal;
            this.invertFlagRotation = invertFlagRotation;
            this.isectX = isectX;
            this.isectY = isectY;
        }

        public Split(String text) {
            List<String> parts = Arrays.asList(text.split("\\$"));
            Iterator<String> it = parts.iterator();
            this.interface1 = readInts(it);
            this.interface2 = readInts(it);
            this.lines2 = readInts(it);
            this.terminals1 = readInts(it);
            this.terminals2 = readInts(it);
            this.isHorizontal = Boolean.parseBoolean(it.next());
            this.invertFlagRotation = Boolean.parseBoolean(it.next());
            this.isectX = Double.parseDouble(it.next());
            this.isectY = Double.parseDouble(it.next());
        }

        @Override
        public String toString() {
            StringBuilder b = new StringBuilder();
            b.append("S");
            write(b, interface1);
            b.append("$");
            write(b, interface2);
            b.append("$");
            write(b, lines2);
            b.append("$");
            write(b, terminals1);
            b.append("$");
            write(b, terminals2);
            b.append("$");
            b.append(isHorizontal);
            b.append("$");
            b.append(invertFlagRotation);
            b.append("$");
            b.append(isectX);
            b.append("$");
            b.append(isectY);
            return b.toString();
        }
    }
    
    private static void write(StringBuilder b, int[] ids) {
        b.append(ids.length);
        for(int e : ids) {
            b.append('$');
            b.append(e);
        }
    }
    
    private static int[] readInts(Iterator<String> it) {
        int length = Integer.parseInt(it.next());
        int[] result = new int[length];
        for(int i=0;i<length;++i)
            result[i] = Integer.parseInt(it.next());
        return result;
    }
    
    public void addModi(Modi modi) {
        modis.add(modi);
    }

    public RouteGraphModification(String text) {
        String[] parts = text.split(",");
        int pos = 0;
        lineCount = Integer.parseInt(parts[pos++]);
        terminalCount = Integer.parseInt(parts[pos++]);
        terminalIdentifiers = new String[terminalCount];
        for(int i=0;i<terminalCount;++i) {
            terminalIdentifiers[i] = parts[pos++];            
        }
        int linkCount = Integer.parseInt(parts[pos++]);
        links = new int[2*linkCount];
        for(int i=0;i<links.length;++i)
            links[i] = Integer.parseInt(parts[pos++]);
        while(pos < parts.length) {
            String part = parts[pos++];
            char first = part.charAt(0);
            part = part.substring(1);
            switch(first) {
            case 'M': addModi(new UpdateLine(part)); break;
            case 'R': addModi(new RemoveLine(part)); break;
            case 'r': addModi(new RemoveLink(part)); break;
            case 'C': addModi(new CreateLine(part)); break;
            case 'c': addModi(new CreateLink(part)); break;
            case 'S': addModi(new Split(part)); break;
            }
        }
        resources = new Resource[lineCount + terminalCount];
    }

    public RouteGraphModification(SerialisationSupport ser, RouteGraph rg) throws DatabaseException {
        Collection<RouteLine> lines = rg.getLines();
        lineCount = lines.size();
        Collection<RouteTerminal> terminals = rg.getTerminals();
        terminalCount = terminals.size();

        resources = new Resource[lineCount + terminalCount];

        THashSet<RouteLink> linkSet = new THashSet<RouteLink>();
        idMap = new TObjectIntHashMap<RouteNode>(); 

        int i=0;
        for(RouteLine line : lines) {
            idMap.put(line, i);
            resources[i] = ser.getResource((Long)line.getData());
            for(RoutePoint rp : line.getPoints()) {
                if(rp instanceof RouteLink) {
                    RouteLink link = (RouteLink)rp;
                    if(!link.getA().isTransient() &&
                            !link.getA().isTransient())
                        linkSet.add(link);
                }
            }
            ++i;
        }
        for(RouteTerminal terminal : terminals) {
            idMap.put(terminal, i);
            resources[i] = ser.getResource((Long)terminal.getData());
            ++i;
        }
        if(rg.isSimpleConnection()) {
            links = new int[] {0, 1};
        }
        else {
            links = new int[2*(terminalCount + linkSet.size())];
            i = 0;        
            for(RouteLink link : linkSet) {
                links[i++] = idMap.get(link.getA());
                links[i++] = idMap.get(link.getB());
            }
            for(RouteTerminal terminal : terminals) {
                links[i++] = idMap.get(terminal);
                links[i++] = idMap.get(terminal.getLine());
            }
        }
    }

    public Resource findTerminalIdentifiers(ReadGraph g) throws DatabaseException {
        Resource base = null;
        terminalIdentifiers = new String[terminalCount];
        for(int i=0;i<terminalCount;++i) {
            Resource r = resources[lineCount + i];
            RelativeReference ref = ElementIdentification.getConnectorIdentifier(g, r);
            terminalIdentifiers[i] = ref.path;
            if(ref.base != null)
                base = ref.base;            
        }
        return base;
    }

    @Override
    public String toString() {
        StringBuilder b = new StringBuilder();
        toString(b);
        return b.toString();
    }

    public void toString(StringBuilder b) {
        b.append(lineCount);
        b.append(',');
        b.append(terminalCount);
        for(int i=0;i<terminalCount;++i) {
            b.append(',');
            b.append(terminalIdentifiers[i]);
        }
        b.append(',');
        b.append(links.length/2);
        for(int l : links) {
            b.append(',');
            b.append(l);
        }
        for(Modi modi : modis) {
            b.append(',');
            b.append(modi);
        }
    }

    public boolean resolveResources(ReadGraph g, Resource model) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(g);
        StructuralResource2 STR = StructuralResource2.getInstance(g);

        // --- Resolve connectors ---------------------------------------------

        Resource connection = null;
        {
            ArrayList<List<Resource>> connectorCandidates = new ArrayList<List<Resource>>(terminalCount); 
            for(int i=0;i<terminalCount;++i)
                connectorCandidates.add(ElementIdentification.resolveConnector(g, model, terminalIdentifiers[i]));            
            for(List<Resource> connectors : connectorCandidates) {
                if(connectors.isEmpty())
                    return false;
                if(connection == null && connectors.size() == 1) {
                    for(Resource temp : g.getObjects(connectors.get(0), STR.Connects))
                        if(g.isInstanceOf(temp, DIA.Connection)) {
                            connection = temp;
                            break;
                        }
                }
            }
            if(connection == null)
                return false;
            loop: for(int i=0;i<terminalCount;++i) {
                for(Resource connector : connectorCandidates.get(i))
                    if(g.hasStatement(connector, STR.Connects, connection)) {
                        resources[lineCount + i] = connector;
                        continue loop;
                    }
                return false;
            }
        }

        if(lineCount != g.getObjects(connection, DIA.HasInteriorRouteNode).size())
            return false;
        if(terminalCount != g.getObjects(connection, STR.IsConnectedTo).size())
            return false;

        if(lineCount == 0)
            return true;

        // --- Resolve route lines --------------------------------------------

        // Create inverse map for resources
        TObjectIntHashMap<Resource> invResources = new TObjectIntHashMap<Resource>();
        for(int i=0;i<terminalCount;++i) {
            int id = lineCount + i;
            invResources.put(resources[id], id);
        }

        // Create neighbor indices
        final TIntHashSet[] neighbors = new TIntHashSet[terminalCount + lineCount];
        for(int i=0;i<neighbors.length;++i)
            neighbors[i] = new TIntHashSet();
        for(int i=0;i<links.length;i+=2) {
            int a = links[i];
            int b = links[i+1];
            if(resources[b] == null)
                neighbors[a].add(b);
            if(resources[a] == null)
                neighbors[b].add(a);
        }

        // Create stack
        TIntArrayList stack = new TIntArrayList();
        TIntArrayList backlog = new TIntArrayList();
        for(int i=0;i<terminalCount;++i)
            stack.add(lineCount+i);

        // Resolve route lines
        int oldResolvedCount = 0;
        while(invResources.size() < resources.length) {
            oldResolvedCount = invResources.size();
            while(!stack.isEmpty()) {
                int id = stack.removeAt(stack.size()-1);
                TIntHashSet ns = neighbors[id];
                for(int n : ns.toArray())
                    if(resources[n] != null)
                        ns.remove(n);
                if(ns.isEmpty())
                    ;
                else if(ns.size() == 1) {
                    Resource det = null;
                    for(Resource r : g.getObjects(resources[id], DIA.AreConnected))
                        if(!invResources.containsKey(r)) {
                            if(det == null)
                                det = r;
                            else
                                return false;
                        }
                    if(det == null)
                        return false;
                    final int newId = ns.iterator().next();
                    resources[newId] = det;
                    invResources.put(det, newId);
                    stack.add(newId);
                }
                else
                    backlog.add(id);
            }

            if(oldResolvedCount == invResources.size())
                return false; // No progress happened

            // Reverse backlog and swap stack and backlog
            {
                backlog.reverse();

                TIntArrayList temp = stack;
                stack = backlog;                
                backlog = temp;
            }
        }

        return true;
    }

    public Resource getConnection(ReadGraph g) throws DatabaseException {        
        if(connection == null) {
            DiagramResource DIA = DiagramResource.getInstance(g);    		
            if(lineCount > 0) {
                connection = g.getSingleObject(resources[0], DIA.HasInteriorRouteNode_Inverse);
            }
            else {
                StructuralResource2 STR = StructuralResource2.getInstance(g);
                for(Resource temp : g.getObjects(resources[0], STR.Connects))
                    if(g.isInstanceOf(temp, DIA.Connection)) {
                        connection = temp;
                        break;
                    }
            }
        }
        return connection;
    }

    public void runUpdates(WriteGraph g) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(g);
        Layer0 L0 = Layer0.getInstance(g);

        for(Modi modi_ : modis) {
            if(modi_ instanceof UpdateLine) {
                UpdateLine modi = (UpdateLine)modi_;
                Resource routeLine = resources[modi.id];
                g.claimLiteral(routeLine, DIA.HasPosition, modi.position);
                g.claimLiteral(routeLine, DIA.IsHorizontal, modi.isHorizontal);
            }
            else if(modi_ instanceof RemoveLink) {
                RemoveLink modi = (RemoveLink)modi_;
                g.denyStatement(resources[modi.a], DIA.AreConnected, resources[modi.b]);
            }
            else if(modi_ instanceof RemoveLine) {
                RemoveLine modi = (RemoveLine)modi_;
                g.deny(resources[modi.a]);
            }
            else if(modi_ instanceof CreateLink) {
                CreateLink modi = (CreateLink)modi_;
                g.claim(resources[modi.a], DIA.AreConnected, resources[modi.b]);
            }
            else if(modi_ instanceof CreateLine) {
                CreateLine modi = (CreateLine)modi_;
                Resource routeLine = g.newResource();
                g.claim(routeLine, L0.InstanceOf, DIA.RouteLine);
                g.claimLiteral(routeLine, DIA.HasPosition, modi.position);
                g.claimLiteral(routeLine, DIA.IsHorizontal, modi.isHorizontal);                        	    
                g.claim(getConnection(g), DIA.HasInteriorRouteNode, routeLine);
                int id = resources.length;
                resources = Arrays.copyOf(resources, id+1);
                resources[id] = routeLine;
            }
            else if(modi_ instanceof Split) {
                Split modi = (Split)modi_;
                RouteGraphConnectionSplitter splitter = new RouteGraphConnectionSplitter(g);
                splitter.doSplit(g, connection,
                        toResources(modi.interface1),
                        toResources(modi.interface2),
                        toResources(modi.lines2),
                        toResources(modi.terminals1),
                        toResources(modi.terminals2),
                        modi.isHorizontal,
                        modi.invertFlagRotation,
                        modi.isectX,
                        modi.isectY
                        );
            }
        }
    }

    public TObjectIntHashMap<RouteNode> getIdMap() {
        return idMap;
    }

    public int[] toIds(ArrayList<Resource> rs) {        
        TObjectIntHashMap<Resource> rmap = new TObjectIntHashMap<Resource>();
        for(int i=0;i<resources.length;++i)
            rmap.put(resources[i], i);
        
        int[] result = new int[rs.size()];
        for(int i=0;i<rs.size();++i)
            result[i] = rmap.get(rs.get(i));
        
        return result;
    }
    
    public ArrayList<Resource> toResources(int[] ids) {
        ArrayList<Resource> result = new ArrayList<Resource>(ids.length);
        for(int id : ids)
            result.add(resources[id]);
        return result;
    }
}
