/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.symbollibrary.ui;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.ACC;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleControlAdapter;
import org.eclipse.swt.accessibility.AccessibleControlEvent;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseTrackListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.internal.WorkbenchImages;
import org.eclipse.ui.internal.WorkbenchMessages;

@SuppressWarnings("restriction")
public class FilterArea extends Composite {

    private final LocalResourceManager resourceManager;

    private Text                       filterText;

    /**
     * The control representing the clear button for the filter text entry. This
     * value may be <code>null</code> if no such button exists, or if the
     * controls have not yet been created.
     */
    protected Control                  clearButtonControl;

    /**
     * Construct the filter area UI component.
     * 
     * @param explorer
     * @param queryProcessor
     * @param parent
     * @param style
     */
    public FilterArea(final Composite parent, int style) {
        super(parent, style | SWT.BORDER);

        resourceManager = new LocalResourceManager(JFaceResources.getResources(parent.getDisplay()), this);

//        GridDataFactory.fillDefaults().grab(true, false).applyTo(this);
//        GridLayoutFactory.fillDefaults().applyTo(this);
//
//        filterText = new Text(this, SWT.BORDER | SWT.SEARCH | SWT.ICON_CANCEL);
//        filterText.setFont(resourceManager.createFont(FontDescriptor.createFrom(filterText.getFont()).increaseHeight(-1)));
//        filterText.setLayoutData(GridDataFactory.fillDefaults().grab(true, true).align(SWT.FILL, SWT.BEGINNING).create());

        GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(this);
        GridLayoutFactory.fillDefaults().margins(0, 0).spacing(0, 0).numColumns(2).equalWidth(false).applyTo(this);

        createFilterText(this);
        createFilterCancelIcon(this);

        this.setBackground(filterText.getBackground());
    }

    private void createFilterText(FilterArea filterArea) {
        //filterText = new Text(this, SWT.SINGLE | SWT.FLAT | SWT.SEARCH | SWT.ICON_CANCEL);
        filterText = new Text(this, SWT.SINGLE | SWT.ICON_CANCEL);
        GridDataFactory.fillDefaults().grab(true, false).applyTo(filterText);

        filterText.setFont(resourceManager.createFont(FontDescriptor.createFrom(filterText.getFont()).increaseHeight(-1)));

        // if we're using a field with built in cancel we need to listen for
        // default selection changes (which tell us the cancel button has been
        // pressed)
        if ((filterText.getStyle() & SWT.ICON_CANCEL) != 0) {
            filterText.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetDefaultSelected(SelectionEvent e) {
                    if (e.detail == SWT.ICON_CANCEL)
                        clearText();
                }
            });
        }

        GridData gridData = new GridData(SWT.FILL, SWT.CENTER, true, true);

        // if the text widget supported cancel then it will have it's own
        // integrated button. We can take all of the space.
        if ((filterText.getStyle() & SWT.ICON_CANCEL) != 0)
            gridData.horizontalSpan = 2;
        filterText.setLayoutData(gridData);
    }

    private void createFilterCancelIcon(Composite parent) {
        // only create the button if the text widget doesn't support one
        // natively
        if ((filterText.getStyle() & SWT.ICON_CANCEL) == 0) {
            final Image inactiveImage = WorkbenchImages.getImage(ISharedImages.IMG_ETOOL_CLEAR_DISABLED);
            final Image activeImage = WorkbenchImages.getImage(ISharedImages.IMG_ETOOL_CLEAR);
            final Image pressedImage = (Image) resourceManager.get( ImageDescriptor.createWithFlags( ImageDescriptor.createFromImage( activeImage ), SWT.IMAGE_GRAY ) );

            final Label clearButton= new Label(parent, SWT.NONE);
            clearButton.setLayoutData(new GridData(SWT.BEGINNING, SWT.CENTER, false, false));
            clearButton.setImage(inactiveImage);
            clearButton.setBackground(parent.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND));
            clearButton.setToolTipText(WorkbenchMessages.FilteredTree_ClearToolTip);
            clearButton.addMouseListener(new MouseAdapter() {
                private MouseMoveListener fMoveListener;

                @Override
                public void mouseDown(MouseEvent e) {
                    clearButton.setImage(pressedImage);
                    fMoveListener= new MouseMoveListener() {
                        private boolean fMouseInButton= true;

                        public void mouseMove(MouseEvent e) {
                            boolean mouseInButton= isMouseInButton(e);
                            if (mouseInButton != fMouseInButton) {
                                fMouseInButton= mouseInButton;
                                clearButton.setImage(mouseInButton ? pressedImage : inactiveImage);
                            }
                        }
                    };
                    clearButton.addMouseMoveListener(fMoveListener);
                }

                @Override
                public void mouseUp(MouseEvent e) {
                    if (fMoveListener != null) {
                        clearButton.removeMouseMoveListener(fMoveListener);
                        fMoveListener= null;
                        boolean mouseInButton= isMouseInButton(e);
                        clearButton.setImage(mouseInButton ? activeImage : inactiveImage);
                        if (mouseInButton) {
                            clearText();
                            filterText.setFocus();
                        }
                    }
                }

                private boolean isMouseInButton(MouseEvent e) {
                    Point buttonSize = clearButton.getSize();
                    return 0 <= e.x && e.x < buttonSize.x && 0 <= e.y && e.y < buttonSize.y;
                }
            });
            clearButton.addMouseTrackListener(new MouseTrackListener() {
                public void mouseEnter(MouseEvent e) {
                    clearButton.setImage(activeImage);
                }

                public void mouseExit(MouseEvent e) {
                    clearButton.setImage(inactiveImage);
                }

                public void mouseHover(MouseEvent e) {
                }
            });
            clearButton.getAccessible().addAccessibleListener(
                    new AccessibleAdapter() {
                        @Override
                        public void getName(AccessibleEvent e) {
                            e.result= WorkbenchMessages.FilteredTree_ClearToolTip;
                        }
                    });
            clearButton.getAccessible().addAccessibleControlListener(
                    new AccessibleControlAdapter() {
                        @Override
                        public void getRole(AccessibleControlEvent e) {
                            e.detail= ACC.ROLE_PUSHBUTTON;
                        }
                    });
            this.clearButtonControl = clearButton;
        }
    }

    /**
     * Clears the text in the filter text widget.
     */
    protected void clearText() {
        filterText.setText(""); //$NON-NLS-1$
        //textChanged();
    }

    public Text getText() {
        return filterText;
    }

    public void focus() {
        if (filterText.isDisposed())
            return;
        Display d = filterText.getDisplay();
        if (d.getThread() == Thread.currentThread()) {
            doSetFocus();
        } else {
            d.asyncExec(new Runnable() {
                @Override
                public void run() {
                    if (filterText.isDisposed())
                        return;
                    doSetFocus();
                }
            });
        }
    }

    protected void doSetFocus() {
        filterText.selectAll();
        filterText.setFocus();
    }

}
