/*******************************************************************************
 * Copyright (c) 2007 VTT Technical Research Centre of Finland and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.participant;

import java.awt.Font;
import java.awt.RenderingHints;
import java.awt.geom.Rectangle2D;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.batik.dom.GenericDOMImplementation;
import org.apache.batik.svggen.SVGGeneratorContext;
import org.apache.batik.svggen.SVGGeneratorContext.GraphicContextDefaults;
import org.apache.batik.svggen.SVGGraphics2D;
import org.apache.batik.svggen.SVGGraphics2DIOException;
import org.apache.batik.util.SVGConstants;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.widgets.Display;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.canvas.impl.DependencyReflection.Dependency;
import org.simantics.g2d.diagram.participant.AbstractDiagramParticipant;
import org.simantics.g2d.diagram.participant.Selection;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.G2DRenderingHints;
import org.simantics.scenegraph.g2d.G2DSceneGraph;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.g2d.events.EventHandlerReflection.EventHandler;
import org.simantics.scenegraph.g2d.events.command.CommandEvent;
import org.simantics.scenegraph.g2d.events.command.Commands;
import org.simantics.scenegraph.g2d.nodes.LinkNode;
import org.simantics.scenegraph.g2d.nodes.SingleElementNode;
import org.simantics.scenegraph.utils.NodeMapper;
import org.simantics.utils.ui.ErrorLogger;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

public class CopyAsSVGParticipant extends AbstractDiagramParticipant {

    @Dependency
    protected Selection sel;

    @EventHandler(priority = 0)
    public boolean handleCommand(CommandEvent e) {
        if (e.command.equals(Commands.COPY_AS_SVG)) {
            Set<IElement> ss = sel.getSelection(0);
            copyAsSVG(getContext(), ss);
            return true;
        }
        return false;
    }

    private static void copyAsSVG(ICanvasContext canvasContext, Set<IElement> elements) {
        G2DSceneGraph sg = canvasContext.getSceneGraph();
        NodeMapper clipboardNodeMapper = new NodeMapper();
        SingleElementNode clipboardNode = sg.addNode("svg-clipboard-temp", SingleElementNode.class); 

        try {
            for (IElement e : elements) {
                INode node = e.getHint(ElementHints.KEY_SG_NODE);
                if (node != null) {
                    String nodeId = clipboardNodeMapper.add(node);
                    LinkNode delegate = clipboardNode.addNode(ElementUtils.generateNodeId(e), LinkNode.class);
                    if (node instanceof IG2DNode g2n) {
                        delegate.setZIndex(g2n.getZIndex());
                    }
                    delegate.setDelegateId( nodeId );
                }
            }

            DOMImplementation domImpl =  GenericDOMImplementation.getDOMImplementation();

            String svgNS = "http://www.w3.org/2000/svg";
            Document document = domImpl.createDocument(svgNS, "svg", null);

            GraphicContextDefaults gcDefaults = new GraphicContextDefaults();
            SVGGeneratorContext ctx = SVGGeneratorContext.createDefault(document);
            Map<java.awt.RenderingHints.Key, Object> hintMap = new HashMap<java.awt.RenderingHints.Key, Object>();

            hintMap.put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
            hintMap.put(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            hintMap.put(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
            hintMap.put(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
            hintMap.put(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);

            gcDefaults.setRenderingHints(new RenderingHints(hintMap));
            gcDefaults.setFont(Font.decode(null));
            ctx.setGraphicContextDefaults(gcDefaults);

            SVGGraphics2D svgG2D = new SVGGraphics2D(ctx, false);

            svgG2D.setRenderingHint(G2DRenderingHints.KEY_HIDE_SELECTION, true);

            StringWriter writer = new StringWriter();

            // Track connection crossings manually since we will render only the clipboard node.
            sg.getNode(ConnectionCrossingsParticipant.CONNECTION_CROSSINGS_NODE_KEY).render(svgG2D);
            clipboardNode.render(svgG2D);

            Element root = svgG2D.getRoot();

            Rectangle2D bounds = clipboardNode.getBoundsInLocal(true);
            if (bounds != null) {
                root.setAttributeNS(null, "viewBox", bounds.getMinX() + " " + bounds.getMinY() + " " + bounds.getWidth() + " " + bounds.getHeight()); 
                root.setAttributeNS(null, "height", Double.toString(bounds.getHeight()));
                root.setAttributeNS(null, "width", Double.toString(bounds.getWidth()));
            }

            // This is a workaround to get copied SVG working with Inkscape.
            // Inkscape loses at least stroke attached to the root <svg> element.
            // This copies a bunch of stroke/fill/font related attributes from
            // the SVG root to all <g> nodes directly under to allow also Inkscape
            // to understand all those attributes.
            copyRootAttributesToRootGroups(root);

            try {
                svgG2D.stream(root, writer, false, false);
            } catch (SVGGraphics2DIOException e1) {
                ErrorLogger.defaultLogError("Failed to copy the diagram selection as SVG." , e1);
            }

            String svgText = writer.toString();
            byte[] svgContent = svgText.getBytes(StandardCharsets.UTF_8);

            Display.getDefault().asyncExec(() -> {
                Clipboard cb = new Clipboard(Display.getCurrent());
                cb.setContents(
                        new Object[]   { svgContent, svgText },
                        new Transfer[] { SVGTransfer.getInstance(), TextTransfer.getInstance() });
            });

        } finally {
            clipboardNode.removeNodes();
            clipboardNodeMapper.clear();
            clipboardNode.remove();
        }
    }

    private static List<String> copiedAttributes = Arrays.asList(
            SVGConstants.SVG_STROKE_ATTRIBUTE,
            SVGConstants.SVG_STROKE_DASHARRAY_ATTRIBUTE,
            SVGConstants.SVG_STROKE_DASHOFFSET_ATTRIBUTE,
            SVGConstants.SVG_STROKE_LINECAP_ATTRIBUTE,
            SVGConstants.SVG_STROKE_LINEJOIN_ATTRIBUTE,
            SVGConstants.SVG_STROKE_MITERLIMIT_ATTRIBUTE,
            SVGConstants.SVG_STROKE_OPACITY_ATTRIBUTE,
            SVGConstants.SVG_STROKE_WIDTH_ATTRIBUTE,
            SVGConstants.SVG_FILL_ATTRIBUTE,
            SVGConstants.SVG_FILL_OPACITY_ATTRIBUTE,
            SVGConstants.SVG_FONT_FAMILY_ATTRIBUTE,
            SVGConstants.SVG_FONT_SIZE_ATTRIBUTE,
            SVGConstants.SVG_FONT_STYLE_ATTRIBUTE,
            SVGConstants.SVG_FONT_WEIGHT_ATTRIBUTE
    );

    private static void copyRootAttributesToRootGroups(Element root) {
        List<String> values = new ArrayList<>(copiedAttributes.size());
        for (String attr : copiedAttributes) {
            values.add(root.getAttribute(attr));
        }

        var rch = root.getChildNodes();
        for (int i = 0; i < rch.getLength(); ++i) {
            var n = rch.item(i);
            if (n.getNodeType() == Node.ELEMENT_NODE && SVGConstants.SVG_G_TAG.equals(n.getNodeName())) {
                var e = (Element) n;
                for (int j = 0; j < values.size(); ++j) {
                    var attr = copiedAttributes.get(j);
                    var v = values.get(j);
                    if (!v.isEmpty() && !e.hasAttribute(attr)) {
                        e.setAttribute(attr, v);
                    }
                }
            }
        }
    }

}
