/*******************************************************************************
 * Copyright (c) 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.handler;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Set;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.primitiverequest.IsInstanceOf;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.connection.RouteGraph;
import org.simantics.diagram.handler.ConnectionSplitAndJoin.Join;
import org.simantics.diagram.handler.ConnectionSplitAndJoin.SplitRouteGraph;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.g2d.canvas.impl.DependencyReflection.Dependency;
import org.simantics.g2d.connection.ConnectionEntity;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.participant.AbstractDiagramParticipant;
import org.simantics.g2d.diagram.participant.Selection;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.elementclass.RouteGraphConnectionClass;
import org.simantics.g2d.participant.MouseUtil;
import org.simantics.g2d.participant.MouseUtil.MouseInfo;
import org.simantics.scenegraph.g2d.events.EventHandlerReflection.EventHandler;
import org.simantics.scenegraph.g2d.events.command.CommandEvent;
import org.simantics.scenegraph.g2d.events.command.Commands;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A participant for handling:
 * <ul>
 * <li>SPLIT_CONNECTION</li>
 * <li>JOIN_FLAGS</li>
 * </ul>
 * 
 * for route graph diagram connections.
 * 
 * @author Tuukka Lehtonen
 * @since 1.60.0
 */
public class RouteGraphConnectionCommandHandler extends AbstractDiagramParticipant {

	private static final Logger LOGGER = LoggerFactory.getLogger(RouteGraphConnectionCommandHandler.class);

	@Dependency MouseUtil mouseUtil;
	@Dependency Selection selection;

	@EventHandler(priority = 100)
	public boolean handleCommand(CommandEvent event) {
		try {
			if (Commands.SPLIT_CONNECTION.equals(event.command)) {
				return splitConnection();
			} else if (Commands.JOIN_FLAGS.equals(event.command)) {
				return joinFlags();
			}
		} catch (DatabaseException e) {
			LOGGER.error("Connection command handling", e);
		}
		return false;
	}

	private boolean splitConnection() throws DatabaseException {
		final IElement connection = getSingleConnection();
		if (connection == null)
			return false;
		final IDiagram diagram = ElementUtils.peekDiagram(connection);
		if (diagram == null)
			return false;
		Object obj = ElementUtils.getData(diagram, connection);
		if (!(obj instanceof Resource))
			return false;

		RouteGraph rg = connection.getHint(RouteGraphConnectionClass.KEY_ROUTEGRAPH);
		if (rg == null)
			return false;

		Session session = Simantics.getSession();
		Resource conn = (Resource) obj;
		DiagramResource DIA = DiagramResource.getInstance(session);
		if (!session.syncRequest(new IsInstanceOf(conn, DIA.RouteGraphConnection)))
			return false;

		MouseInfo mi = mouseUtil.getMouseInfo(0);
		new SplitRouteGraph(session, getContext(), mi.canvasPosition, conn, rg).run();
		return true;
	}

	/**
	 * @return the selected top-level connection element if and only if the
	 *         selection contains the connection or parts of the same connection
	 *         (edge segments or branch/route points) and nothing else.
	 */
	private IElement getSingleConnection() {
		Set<IElement> ss = selection.getSelection(0);
		if (ss.isEmpty())
			return null;

		IElement result = null;

		for (IElement e : ss) {
			ConnectionEntity ce = e.getHint(ElementHints.KEY_CONNECTION_ENTITY);
			if (ce == null)
				continue;
			if (result != null && !ce.getConnection().equals(result))
				return null;
			result = ce.getConnection();
		}

		return result;
	}

	private boolean joinFlags() throws DatabaseException {
		Resource[] sel = getSelectedResources();
		if (sel.length == 0)
			return false;

		Session session = Simantics.getSession();
		Collection<Resource> flags = session.syncRequest(
				(ReadGraph graph) -> ConnectionSplitAndJoin.getLocalConnectedFlags(graph, sel));
		if (flags.isEmpty())
			return false;

		new Join(session, getContext(), flags).run();
		return true;
	}

	private Resource[] getSelectedResources() {
		Set<IElement> ss = selection.getSelection(0);
		if (ss.isEmpty())
			return Resource.NONE;

		var result = new ArrayList<Resource>(ss.size());
		for (var el : ss) {
			Object obj = ElementUtils.getObject(el);
			if (obj instanceof Resource) {
				result.add((Resource) obj);
			}
		}

		return result.toArray(Resource.NONE);
	}

}
