/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.request;

import org.simantics.db.ReadGraph;
import org.simantics.db.Session;
import org.simantics.db.exception.CancelTransactionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.SyncMultiProcedure;

/**
 * The <code>GraphRequest</code> interface is used to create transaction
 * requests to Simantics database implementations. Both read and write
 * transaction requests use the same interface.
 * 
 * <p>
 * The actual work carried out by the implemented request should be done in the
 * <code>perform</code> method. It receives a <code>Graph</code> instance as
 * the only argument which is the interface for actually reading and writing the
 * graph data model.
 * 
 * <p>
 * Transaction requests can be made to the database by creating your own
 * <code>GraphRequest</code> instance and putting it in the request queue of
 * the database session through the {@link Session} interface. The database
 * session is responsible for executing the queued requests in a thread of its
 * choice, or possibly/preferably multiple threads. The database session can
 * allow multiple read-only requests to occur simultaneously, but read-write
 * requests require exclusive database access. In other words only one
 * read-write request can be in execution simultaneously.
 * 
 * <p>
 * This interface also has two callbacks - <code>handleException</code> for
 * allowing handling any exceptions thrown by <code>perform</code> and
 * <code>requestCompleted</code> for performing actions after a request has
 * been successfully completed.
 * 
 * <p>
 * Clients of this interface are encouraged to extend the provided abstract
 * implementations or this class or extend their own helper implementations for
 * ones particular needs. The provided abstract implementations are:
 * <ul>
 * <li>{@link ReadGraphRequestAdapter} provides default implementations for
 * everything except {@link #perform(ReadGraph)}.</li>
 * <li>{@link SimpleGraphRequest} replaces {@link #perform(ReadGraph)} with
 * {@link SimpleGraphRequest#run(ReadGraph)} for easier request implementation in
 * simple cases.</li>
 * <li>{@link GraphRequestWithResult} makes it easier for the user to return
 * a single result Object from the request.</li>
 * </ul>
 * 
 * @author Tuukka Lehtonen
 * @see ReadGraphRequestAdapter
 * @see GraphRequestRunner
 * @see GraphRequestStatus
 * @see GraphRequestWithResult
 * @see Session
 * @see SimpleGraphRequest
 */
public interface MultiRead<Result> {

    /**
     * When a <code>GraphRequest</code> is serviced by the database session
     * the method <code>perform</code> is invoked.
     * 
     * <p>
     * Perform receives an object instance implementing the <code>SyncReadGraph</code>
     * interface which provides the only way to read/write the graph data model.
     * The Graph instance must only be valid during the execution of the
     * <code>perform</code> method and therefore should not be stored for use
     * outside of its execution.
     * 
     * <p>
     * The general contract of the method <code>perform</code> is that it may
     * take any action whatsoever which involves reading or writing the data
     * model through the received Graph instance.
     * 
     * @param g an interface for reading and writing the data model
     * @return the result status of the request which affects how the
     *         transaction proceeds, see GraphRequestStatus for more information
     * @throws Exception when anything goes wrong inside the request thread
     * @throws CancelTransactionException to indicate that the request needs to
     *         be cancelled and any changes rolled back
     */
    void perform(ReadGraph graph, SyncMultiProcedure<Result> callback) throws DatabaseException;
    
 }
