/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db;

import org.simantics.databoard.binding.Binding;
import org.simantics.db.exception.BindingException;
import org.simantics.db.exception.ClusterSetExistException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.request.AsyncMultiRead;
import org.simantics.db.request.Write;
import org.simantics.db.request.WriteOnly;
import org.simantics.db.request.WriteTraits;


/**
 * For accessing and manipulating the graph data.
 * 
 * Instantiated by Session for use only during a transaction.
 * 
 * @author Antti Villberg
 * @see AsyncMultiRead
 * @see Resource
 * @see Statement
 */
public interface WriteOnlyGraph extends ServiceLocator, MetadataI {

    Resource getRootLibrary();
    Resource getDefaultMutableClusterSet();
    Resource getDefaultImmutableClusterSet();

    /**
     * Gets a builtin resource. For a list of builtin resources see TODO Wiki
     * 
     * @param uri the identifier
     * @return the resource
     * @throws ResourceNotFoundException if resouce was not found
     * @throws ServiceException on connection and database failures
     * @see AsyncReadGraph#forBuiltin
     */
    Resource getBuiltin(String id) throws ResourceNotFoundException, ServiceException;

    /**
     * Creates a new resource.
     * @throws ServiceException
     * 
     * @returns the new resource
     */
    Resource newResource() throws ServiceException;

    /**
     * Creates a resource into the specified cluster.
     * 
     * @param clusterId the ID of the cluster to create the resource into
     * @throws ServiceException
     * @returns the new resource
     */
    Resource newResource(long clusterId) throws ServiceException;

    /**
     * Creates a resource into the specified cluster set.
     * 
     * @param clusterSet identifies the cluster set which must exist.
     * @throws ServiceException 
     * @throws ClusterSetExistException if cluster set does not exist.
     * @returns the new resource
     * @see #newClusterSet
     */
    Resource newResource(Resource clusterSet) throws ServiceException, ClusterSetExistException;

    /**
     * Creates a new cluster set.
     * 
     * @param clusterSet identifies the created cluster set.
     * @throws ServiceException
     * @throws ClusterSetExistException if cluster set exists already.
     */
    void newClusterSet(Resource clusterSet) throws ServiceException, ClusterSetExistException;
    /**
     * Sets cluster set to be used for {@link #newResource()} method.
     * If cluster set is null then newResource behaves as before. 
     * 
     * @param clusterSet
     * @return old cluster set.
     * @throws ServiceException
     */
    Resource setClusterSet4NewResource(Resource clusterSet) throws ServiceException;
    
    /**
     * Makes sure that the statements (s,p,o) and (o,i,s) are found in the
     * graph. It is not asserted that p is an inverse relation of i, this is
     * assumed and considered to be the client's responsibility.
     * 
     * <p>
     * If no inverse relation (o,i,s) is desired, <code>null</code> may be given as inverse.
     * </p>
     * 
     * @param subject
     * @param predicate
     * @param inverse
     * @param object
     */
    void claim(Resource subject, Resource predicate, Resource inverse, Resource object) throws ServiceException;

    /**
     * Stores the specified literal value (boxed primitive/String or
     * primitive/String array) into the specified resource. Any previous value
     * is overridden.
     * 
     * @param resource the resource to store the literal value into
     * @param value the literal value
     */
    void claimValue(Resource resource, Object value) throws ServiceException;
    void claimValue(Resource resource, Object value, Binding binding) throws ServiceException;

    void addLiteral(Resource resource, Resource predicate, Resource inverse, Object value, Binding binding) throws BindingException, ManyObjectsForFunctionalRelationException, ServiceException;
    void addLiteral(Resource resource, Resource predicate, Resource inverse, Resource type, Object value, Binding binding) throws BindingException, ManyObjectsForFunctionalRelationException, ServiceException;

    /**
     * Makes sure that the statements (s,p,o) and (o,i,s) are no longer found in
     * the graph. It is not asserted that p is an inverse relation of i, this is
     * assumed and considered to be the client's responsibility.
     * 
     * @param subject
     * @param predicate
     * @param inverse
     * @param object
     */
    void deny(Resource subject, Resource predicate, Resource inverse, Resource object, VirtualGraph graph) throws ServiceException;

    /**
     * If there is a literal value associated with the specified resource, it is
     * removed. Otherwise does nothing.
     * 
     * @param resource
     */
    void denyValue(Resource resource) throws ServiceException;
    void denyValue(Resource resource, VirtualGraph graph) throws ServiceException;

    /**
     * Ends writing into the current cluster and starts a new one. Any resources
     * created with {@link #newResource()} after this call will go into the new
     * cluster.
     */
    void flushCluster() throws ServiceException;

    /**
     * Flush all modified data in the given cluster stream to server.
     * 
     * @param r is resource used to identify cluster. If null then all clusters
     *        streams are flushed.
     */
    void flushCluster(Resource r) throws ServiceException;

    /**
     * Synchronously performs the given {@link WriteOnly}.
     * 
     * @param request an instance of {@link Write}.
     */
    void syncRequest(WriteOnly r) throws DatabaseException;

    int thread();

    VirtualGraph getProvider();
    
    /**
     * After current request is processed the undo list is cleared.
     */
    void clearUndoList(WriteTraits writeTraits);
    
    /*
     * Returns the immutability status of the resource as defined by its cluster. 
     */
    boolean isImmutable(Resource resource) throws DatabaseException;
    
    /*
     * Returns the immutability status of the resource as defined by its cluster.
     * If service mode allows writing to immutable resources this returns false.
     */
    boolean isImmutableForWriting(Resource resource) throws DatabaseException;

    /**
     * Marks the beginning of the ongoing write transaction as an undo point.
     * Calling this method several times during the same transaction has no
     * effect.
     */
    void markUndoPoint();

}
