/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db;

import java.util.Collection;
import java.util.function.Consumer;

import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncListener;
import org.simantics.db.procedure.AsyncMultiListener;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.Listener;
import org.simantics.db.procedure.MultiListener;
import org.simantics.db.procedure.MultiProcedure;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.procedure.SyncListener;
import org.simantics.db.procedure.SyncMultiListener;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.procedure.SyncProcedure;
import org.simantics.db.request.AsyncMultiRead;
import org.simantics.db.request.AsyncRead;
import org.simantics.db.request.DelayedWrite;
import org.simantics.db.request.DelayedWriteResult;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.Read;
import org.simantics.db.request.Write;
import org.simantics.db.request.WriteOnly;
import org.simantics.db.request.WriteOnlyResult;
import org.simantics.db.request.WriteResult;

/**
 * 
 * For initiating requests in synchronous manner. The possible requests are
 * <ul>
 * <li>{@link Read} for computing a single result synchronously
 * <li>{@link AsyncRead} for computing a single result asynchronously
 * <li>{@link MultiRead} for computing a collection of results synchronously
 * <li>{@link AsyncMultiRead} for computing a collection of results
 * asynchronously
 * <li>{@link Write} for reading and writing synchronously
 * <li>{@link WriteOnly} for writing synchronously
 * </ul>
 * <p>
 * The standard RequestProcessors are
 * <ul>
 * <li>{@link ReadGraph} for performing requests during other requests
 * <li>{@link Session} for initiating a transactions for performing requests
 * <li>{@link MergingGraphRequestProcessor} for merging several requests in a
 * single transaction
 * </ul>
 * 
 * Database services (see e.g. {@link Session}) are available from implemented
 * {@link ServiceLocator}. For an asynchronous counterpart with the same
 * functionality as RequestProcessor see {@link AsyncRequestProcessor}
 * 
 * <p>
 * <b>IMPORTANT:</b>A client invoking any of the <code>syncRequest</code>
 * -methods in this interface must not assume that the request is performed
 * within the same thread that performed the <code>syncRequest</code>
 * invocation. This is an implementation-specific matter.
 * 
 * @version 0.7
 * @author Antti Villberg
 * @see Read
 * @see AsyncRead
 * @see MultiRead
 * @see AsyncMultiRead
 * @see Write
 * @see WriteOnly
 * @see ReadGraph
 * @see Session
 * @see MergingGraphRequestProcessor
 * @see AsyncRequestProcessor
 */
public interface RequestProcessorSpecific {

    /**
     * Synchronously determines and returns the result of the given {@link Read}
     * and registers the given {@link Read} as a dependency of the ongoing
     * request if any. This is usually invoked for the side-effects of the given
     * {@link Read} but can also be used for e.g. dependency registering or
     * cache priming.
     * 
     * @param request an instance of {@link Read}
     * @return the result of the given {@link Read}
     * @throws DatabaseException from {@link Read#perform(ReadGraph)} or from
     *         request processing system
     */
    <T> T syncRequest(Read<T> request) throws DatabaseException;

    /**
     * Synchronously registers the given {@link AsyncListener} (as identified
     * by {@link Object#equals(Object)}) to track changes in the result
     * determined from the given {@link Read}. Registers the given {@link Read}
     * as a dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link AsyncListener}.
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link AsyncListener}.
     */
    //<T> T syncRequest(Read<T> request, AsyncListener<T> procedure) throws DatabaseException;

    /**
     * Synchronously registers the given {@link SyncListener} (as identified by
     * {@link Object#equals(Object)}) to track changes in the result determined
     * from the given {@link Read}. Registers the given {@link Read} as a
     * dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link SyncListener}
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link SyncListener}.
     */
    <T> T syncRequest(Read<T> request, SyncListener<T> procedure) throws DatabaseException;

    /**
     * Synchronously registers the given {@link Listener} (as identified by
     * {@link Object#equals(Object)}) to track changes in the result determined
     * from the given {@link Read}. Registers the given {@link Read} as a
     * dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link Listener}
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link Listener}.
     */
    <T> T syncRequest(Read<T> request, Listener<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given {@link Read}
     * to the given {@link AsyncProcedure}. Registers the given {@link Read} as
     * a dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link AsyncProcedure}
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link AsyncProcedure}.
     */
    <T> T syncRequest(Read<T> request, AsyncProcedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given {@link Read}
     * to the given {@link SyncProcedure}. Registers the given {@link Read} as a
     * dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link SyncProcedure}
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link SyncProcedure}.
     */
    <T> T syncRequest(Read<T> request, SyncProcedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given {@link Read}
     * to the given {@link Procedure}. Registers the given {@link Read} as a
     * dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link Procedure}
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link Procedure}.
     */
    <T> T syncRequest(Read<T> request, Procedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously determines the result of the given {@link AsyncRead} and
     * registers the given {@link AsyncRead} as a dependency of the ongoing
     * request if any. This is usually invoked for the side-effects of the given
     * {@link AsyncRead} but can also be used for e.g. dependency registering or
     * cache priming.
     * 
     * @param request an instance of {@link AsyncRead}.
     * @return the result of the given {@link AsyncRead}
     * @throws DatabaseException from {@link AsyncRead#perform(AsyncReadGraph)} or from
     *         request processing system
     */
    <T> T syncRequest(AsyncRead<T> request) throws DatabaseException;

    /**
     * Synchronously registers the given {@link AsyncListener} (as identified
     * by {@link Object#equals(Object)}) to track changes in the result
     * determined from the given {@link AsyncRead}. Registers the given
     * {@link AsyncRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link AsyncRead} will be transferred to the given {@link AsyncListener}.
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link AsyncListener}.
     */
    <T> T syncRequest(AsyncRead<T> request, AsyncListener<T> procedure) throws DatabaseException;

    /**
     * Synchronously registers the given {@link SyncListener} (as identified by
     * {@link Object#equals(Object)}) to track changes in the result determined
     * from the given {@link AsyncRead}. Registers the given {@link AsyncRead}
     * as a dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link AsyncRead} will be
     * transferred to the given {@link SyncListener}
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link SyncListener}.
     */
    <T> T syncRequest(AsyncRead<T> request, SyncListener<T> procedure) throws DatabaseException;

    /**
     * Synchronously registers the given {@link Listener} (as identified by
     * {@link Object#equals(Object)}) to track changes in the result determined
     * from the given {@link AsyncRead}. Registers the given {@link AsyncRead}
     * as a dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link AsyncRead} will be
     * transferred to the given {@link Listener}
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link Listener}.
     */
    <T> T syncRequest(AsyncRead<T> request, Listener<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given
     * {@link AsyncRead} to the given {@link AsyncProcedure}. Registers the
     * given {@link AsyncRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link AsyncRead} will be transferred to the given {@link AsyncProcedure}
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link AsyncProcedure}.
     */
    <T> T syncRequest(AsyncRead<T> request, AsyncProcedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given
     * {@link AsyncRead} to the given {@link SyncProcedure}. Registers the given
     * {@link AsyncRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link AsyncRead} will be transferred to the given {@link SyncProcedure}
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link SyncProcedure}.
     */
    <T> T syncRequest(AsyncRead<T> request, SyncProcedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given
     * {@link AsyncRead} to the given {@link Procedure}. Registers the given
     * {@link AsyncRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link AsyncRead} will be transferred to the given {@link Procedure}
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link Procedure}.
     */
    <T> T syncRequest(AsyncRead<T> request, Procedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously determines the result of the given {@link MultiRead} and
     * registers the given {@link MultiRead} as a dependency of the ongoing
     * request if any. This is usually invoked for the side-effects of the given
     * {@link MultiRead} but can also be used for e.g. dependency registering or
     * cache priming. All exceptions from request processing will be lost.
     * 
     * @param request an instance of {@link MultiRead}.
     * @return the result of the given {@link MultiRead}
     * @throws DatabaseException from {@link MultiRead#perform(ReadGraph)} or from
     *         request processing system
     */
    <T> Collection<T> syncRequest(MultiRead<T> request) throws DatabaseException;

    /**
     * Synchronously registers the given {@link AsyncMultiListener} (as
     * identified by {@link Object#equals(Object)}) to track changes in the
     * result determined from the given {@link MultiRead}. Registers the given
     * {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given
     * {@link AsyncMultiListener}.
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link AsyncMultiListener}.
     */
    //<T> Collection<T> syncRequest(MultiRead<T> request, AsyncMultiListener<T> procedure) throws DatabaseException;

    /**
     * Synchronously registers the given {@link SyncMultiListener} (as
     * identified by {@link Object#equals(Object)}) to track changes in the
     * result determined from the given {@link MultiRead}. Registers the given
     * {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given
     * {@link SyncMultiListener}
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link SyncMultiListener}.
     */
    <T> Collection<T> syncRequest(MultiRead<T> request, SyncMultiListener<T> procedure) throws DatabaseException;

    /**
     * Synchronously registers the given {@link MultiListener} (as identified
     * by {@link Object#equals(Object)}) to track changes in the result
     * determined from the given {@link MultiRead}. Registers the given
     * {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given {@link MultiListener}
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link MultiListener}.
     */
    <T> Collection<T> syncRequest(MultiRead<T> request, MultiListener<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given
     * {@link MultiRead} to the given {@link AsyncMultiProcedure}. Registers the
     * given {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given
     * {@link AsyncMultiProcedure}
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link AsyncMultiProcedure}.
     */
    //<T> Collection<T> syncRequest(MultiRead<T> request, AsyncMultiProcedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given
     * {@link MultiRead} to the given {@link SyncMultiProcedure}. Registers the
     * given {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given
     * {@link SyncMultiProcedure}
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link SyncMultiProcedure}.
     */
    <T> Collection<T> syncRequest(MultiRead<T> request, SyncMultiProcedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given
     * {@link MultiRead} to the given {@link MultiProcedure}. Registers the
     * given {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given {@link MultiProcedure}
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link MultiProcedure}.
     */
    <T> Collection<T> syncRequest(MultiRead<T> request, MultiProcedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously determines the result of the given {@link AsyncMultiRead}
     * and registers the given {@link AsyncMultiRead} as a dependency of the
     * ongoing request if any. This is usually invoked for the side-effects of
     * the given {@link AsyncMultiRead} but can also be used for e.g. dependency
     * registering or cache priming. All exceptions from request processing will
     * be lost.
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @throws DatabaseException from {@link AsyncMultiRead#perform(AsyncReadGraph)} or from
     *         request processing system
     */
    <T> Collection<T> syncRequest(AsyncMultiRead<T> request) throws DatabaseException;

    /**
     * Synchronously registers the given {@link AsyncMultiListener} (as
     * identified by {@link Object#equals(Object)}) to track changes in the
     * result determined from the given {@link AsyncMultiRead}. Registers the
     * given {@link AsyncMultiRead} as a dependency of the ongoing request if
     * any. Exceptions due to the determination of the result of the given
     * {@link AsyncMultiRead} will be transferred to the given
     * {@link AsyncMultiListener}.
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link AsyncMultiListener}.
     */
    <T> Collection<T> syncRequest(AsyncMultiRead<T> request, AsyncMultiListener<T> procedure) throws DatabaseException;

    /**
     * Synchronously registers the given {@link SyncMultiListener} (as
     * identified by {@link Object#equals(Object)}) to track changes in the
     * result determined from the given {@link AsyncMultiRead}. Registers the
     * given {@link AsyncMultiRead} as a dependency of the ongoing request if
     * any. Exceptions due to the determination of the result of the given
     * {@link AsyncMultiRead} will be transferred to the given
     * {@link SyncMultiListener}
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link SyncMultiListener}.
     */
    <T> Collection<T> syncRequest(AsyncMultiRead<T> request, SyncMultiListener<T> procedure) throws DatabaseException;

    /**
     * Synchronously registers the given {@link MultiListener} (as identified
     * by {@link Object#equals(Object)}) to track changes in the result
     * determined from the given {@link AsyncMultiRead}. Registers the given
     * {@link AsyncMultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link AsyncMultiRead} will be transferred to the given
     * {@link MultiListener}
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link MultiListener}.
     */
    <T> Collection<T> syncRequest(AsyncMultiRead<T> request, MultiListener<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given
     * {@link AsyncMultiRead} to the given {@link AsyncMultiProcedure}.
     * Registers the given {@link AsyncMultiRead} as a dependency of the ongoing
     * request if any. Exceptions due to the determination of the result of the
     * given {@link AsyncMultiRead} will be transferred to the given
     * {@link AsyncMultiProcedure}
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link AsyncMultiProcedure}.
     */
    <T> Collection<T> syncRequest(AsyncMultiRead<T> request, AsyncMultiProcedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given
     * {@link AsyncMultiRead} to the given {@link SyncMultiProcedure}. Registers
     * the given {@link AsyncMultiRead} as a dependency of the ongoing request
     * if any. Exceptions due to the determination of the result of the given
     * {@link AsyncMultiRead} will be transferred to the given
     * {@link SyncMultiProcedure}
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link SyncMultiProcedure}.
     */
    <T> Collection<T> syncRequest(AsyncMultiRead<T> request, SyncMultiProcedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously supplies the result determined from the given
     * {@link AsyncMultiRead} to the given {@link MultiProcedure}. Registers the
     * given {@link AsyncMultiRead} as a dependency of the ongoing request if
     * any. Exceptions due to the determination of the result of the given
     * {@link AsyncMultiRead} will be transferred to the given
     * {@link MultiProcedure}
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link MultiProcedure}.
     */
    <T> Collection<T> syncRequest(AsyncMultiRead<T> request, MultiProcedure<T> procedure) throws DatabaseException;


    <T> T syncRequest(ExternalRead<T> request) throws DatabaseException;
    <T> T syncRequest(ExternalRead<T> request, Listener<T> procedure) throws DatabaseException;
    <T> T syncRequest(ExternalRead<T> request, Procedure<T> procedure) throws DatabaseException;

    /**
     * Synchronously performs the given {@link Write}.
     * 
     * @param request an instance of {@link Write}.
     */
    void syncRequest(Write request) throws DatabaseException;

    /**
     * Synchronously performs the given {@link DelayedWrite}.
     * 
     * @param request an instance of {@link DelayedWrite}.
     */
    void syncRequest(DelayedWrite request) throws DatabaseException;

    /**
     * Synchronously performs the given {@link WriteOnly}.
     * 
     * @param request an instance of {@link Write}.
     */
    void syncRequest(WriteOnly r) throws DatabaseException;

    /**
     * Synchronously performs the given {@link Write}.
     * 
     * @param request an instance of {@link Write}.
     */
    <T> T syncRequest(WriteResult<T> request) throws DatabaseException;

    /**
     * Synchronously performs the given {@link DelayedWrite}.
     * 
     * @param request an instance of {@link DelayedWrite}.
     */
    <T> T syncRequest(DelayedWriteResult<T> request) throws DatabaseException;

    /**
     * Synchronously performs the given {@link WriteOnly}.
     * 
     * @param request an instance of {@link Write}.
     */
    <T> T syncRequest(WriteOnlyResult<T> r) throws DatabaseException;

    /**
     * Asynchronously performs the given {@link Write}. The outcome of the
     * request will be lost.
     * 
     * @param request an instance of {@link Write}.
     */
    void asyncRequest(Write request);

    /**
     * Asynchronously performs the given {@link Write}. The outcome of the
     * request will be reported to given {@link Consumer} in the form of a
     * DatabaseException raised during request processing or null upon success.
     * 
     * @param request an instance of {@link Write}.
     * @param request an instance of {@link Consumer}.
     */
    void asyncRequest(Write request, Consumer<DatabaseException> callback);

    <T> void asyncRequest(WriteResult<T> r, Procedure<T> procedure);


    /**
     * Asynchronously performs the given {@link WriteOnly}. The outcome of the
     * request will be lost.
     * 
     * @param request an instance of {@link Write}.
     */
    void asyncRequest(DelayedWrite request);

    /**
     * Asynchronously performs the given {@link WriteOnly}. The outcome of the
     * request will be reported to given {@link Consumer} in the form of a
     * DatabaseException raised during request processing or null upon success.
     * 
     * @param request an instance of {@link WriteOnly}.
     * @param request an instance of {@link Consumer}.
     */
    void asyncRequest(DelayedWrite request, Consumer<DatabaseException> callback);

    <T> void asyncRequest(DelayedWriteResult<T> r, Procedure<T> procedure);

    /**
     * Asynchronously performs the given {@link WriteOnly}. The outcome of the
     * request will be lost.
     * 
     * @param request an instance of {@link Write}.
     */
    void asyncRequest(WriteOnly r);

    /**
     * Asynchronously performs the given {@link WriteOnly}. The outcome of the
     * request will be reported to given {@link Consumer} in the form of a
     * DatabaseException raised during request processing or null upon success.
     * 
     * @param request an instance of {@link WriteOnly}.
     * @param request an instance of {@link Consumer}.
     */
    void asyncRequest(WriteOnly r, Consumer<DatabaseException> callback);

    <T> void asyncRequest(WriteOnlyResult<T> r, Procedure<T> procedure);

}
