/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db;

import java.util.function.Consumer;

import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncListener;
import org.simantics.db.procedure.AsyncMultiListener;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.Listener;
import org.simantics.db.procedure.MultiListener;
import org.simantics.db.procedure.MultiProcedure;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.procedure.SyncListener;
import org.simantics.db.procedure.SyncMultiListener;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.procedure.SyncProcedure;
import org.simantics.db.request.AsyncMultiRead;
import org.simantics.db.request.AsyncRead;
import org.simantics.db.request.DelayedWrite;
import org.simantics.db.request.DelayedWriteResult;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.Read;
import org.simantics.db.request.Write;
import org.simantics.db.request.WriteOnly;
import org.simantics.db.request.WriteOnlyResult;
import org.simantics.db.request.WriteResult;

/**
 * 
 * For initiating requests in asynchronous manner. The possible requests are
 * <ul>
 * <li>{@link Read} for computing a single result synchronously
 * <li>{@link AsyncRead} for computing a single result asynchronously
 * <li>{@link MultiRead} for computing a collection of results synchronously
 * <li>{@link AsyncMultiRead} for computing a collection of results
 * asynchronously
 * <li>{@link Write} for reading and writing synchronously
 * <li>{@link WriteOnly} for writing synchronously
 * </ul>
 * <p>
 * The standard AsyncRequestProcessors are
 * <ul>
 * <li>{@link AsyncReadGraph} for performing requests during other requests
 * <li>{@link Session} for initiating a transactions for performing requests
 * <li>{@link MergingGraphRequestProcessor} for merging several requests in a
 * single transaction
 * </ul>
 * 
 * Database services (see e.g. {@link Session}) are available from implemented
 * {@link ServiceLocator}. For a synchronous counterpart with the same
 * functionality as AsyncRequestProcessor see {@link RequestProcessor}
 * 
 * @version 0.7
 * @author Antti Villberg
 * @see Read
 * @see AsyncRead
 * @see MultiRead
 * @see AsyncMultiRead
 * @see Write
 * @see WriteOnly
 * @see AsyncReadGraph
 * @see Session
 * @see MergingGraphRequestProcessor
 * @see RequestProcessor
 */
public interface AsyncRequestProcessorSpecific extends ServiceLocator {

    /**
     * Provides access to a set of standard resources defined in Layer0. The
     * existence of {@link Builtins} is guaranteed at {@link Session} creation
     * time hence the method does not throw. It is also assumed that the
     * identify of the standard resources cannot change during sessions. To work
     * this requires that Builtins.class has been registered with this
     * ServiceLocator. Therefore this is merely a shorthand for
     * <code>ServiceLocator.getService(Builtins.class)</code>.
     * 
     * @deprecated in favor of org.simantics.layer0.Layer0. Will be removed in Simantics 1.2
     * 
     * @return the <code>Builtins</code> instance associated with this session.
     */
//    @Deprecated
//    Builtins getBuiltins();

    /**
     * Asynchronously determines the result of the given {@link Read} and
     * registers the given {@link Read} as a dependency of the ongoing request
     * if any. This is usually invoked for the side-effects of the given
     * {@link Read} but can also be used for e.g. dependency registering or
     * cache priming. All exceptions from request processing will be lost.
     * 
     * @param request an instance of {@link Read}.
     */
    <T> void asyncRequest(Read<T> request);

    /**
     * Asynchronously registers the given {@link AsyncListener} (as identified
     * by {@link Object#equals(Object)}) to track changes in the result
     * determined from the given {@link Read}. Registers the given {@link Read}
     * as a dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link AsyncListener}.
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link AsyncListener}.
     */
    <T> void asyncRequest(Read<T> request, AsyncListener<T> procedure);

    /**
     * Asynchronously registers the given {@link SyncListener} (as identified by
     * {@link Object#equals(Object)}) to track changes in the result determined
     * from the given {@link Read}. Registers the given {@link Read} as a
     * dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link SyncListener}
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link SyncListener}.
     */
    <T> void asyncRequest(Read<T> request, SyncListener<T> procedure);

    /**
     * Asynchronously registers the given {@link Listener} (as identified by
     * {@link Object#equals(Object)}) to track changes in the result determined
     * from the given {@link Read}. Registers the given {@link Read} as a
     * dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link Listener}
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link Listener}.
     */
    <T> void asyncRequest(Read<T> request, Listener<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given {@link Read}
     * to the given {@link AsyncProcedure}. Registers the given {@link Read} as
     * a dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link AsyncProcedure}
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link AsyncProcedure}.
     */
    <T> void asyncRequest(Read<T> request, AsyncProcedure<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given {@link Read}
     * to the given {@link SyncProcedure}. Registers the given {@link Read} as a
     * dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link SyncProcedure}
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link SyncProcedure}.
     */
    <T> void asyncRequest(Read<T> request, SyncProcedure<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given {@link Read}
     * to the given {@link Procedure}. Registers the given {@link Read} as a
     * dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link Read} will be transferred
     * to the given {@link Procedure}
     * 
     * @param request an instance of {@link Read}.
     * @param procedure an instance of {@link Procedure}.
     */
    <T> void asyncRequest(Read<T> request, Procedure<T> procedure);

    /**
     * Asynchronously determines the result of the given {@link AsyncRead} and
     * registers the given {@link AsyncRead} as a dependency of the ongoing
     * request if any. This is usually invoked for the side-effects of the given
     * {@link AsyncRead} but can also be used for e.g. dependency registering or
     * cache priming. All exceptions from request processing will be lost.
     * 
     * @param request an instance of {@link AsyncRead}.
     */
    <T> void asyncRequest(AsyncRead<T> request);

    /**
     * Asynchronously registers the given {@link AsyncListener} (as identified
     * by {@link Object#equals(Object)}) to track changes in the result
     * determined from the given {@link AsyncRead}. Registers the given
     * {@link AsyncRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link AsyncRead} will be transferred to the given {@link AsyncListener}.
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link AsyncListener}.
     */
    <T> void asyncRequest(AsyncRead<T> request, AsyncListener<T> procedure);

    /**
     * Asynchronously registers the given {@link SyncListener} (as identified by
     * {@link Object#equals(Object)}) to track changes in the result determined
     * from the given {@link AsyncRead}. Registers the given {@link AsyncRead}
     * as a dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link AsyncRead} will be
     * transferred to the given {@link SyncListener}
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link SyncListener}.
     */
    <T> void asyncRequest(AsyncRead<T> request, SyncListener<T> procedure);

    /**
     * Asynchronously registers the given {@link Listener} (as identified by
     * {@link Object#equals(Object)}) to track changes in the result determined
     * from the given {@link AsyncRead}. Registers the given {@link AsyncRead}
     * as a dependency of the ongoing request if any. Exceptions due to the
     * determination of the result of the given {@link AsyncRead} will be
     * transferred to the given {@link Listener}
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link Listener}.
     */
    <T> void asyncRequest(AsyncRead<T> request, Listener<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given
     * {@link AsyncRead} to the given {@link AsyncProcedure}. Registers the
     * given {@link AsyncRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link AsyncRead} will be transferred to the given {@link AsyncProcedure}
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link AsyncProcedure}.
     */
    <T> void asyncRequest(AsyncRead<T> request, AsyncProcedure<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given
     * {@link AsyncRead} to the given {@link SyncProcedure}. Registers the given
     * {@link AsyncRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link AsyncRead} will be transferred to the given {@link SyncProcedure}
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link SyncProcedure}.
     */
    <T> void asyncRequest(AsyncRead<T> request, SyncProcedure<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given
     * {@link AsyncRead} to the given {@link Procedure}. Registers the given
     * {@link AsyncRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link AsyncRead} will be transferred to the given {@link Procedure}
     * 
     * @param request an instance of {@link AsyncRead}.
     * @param procedure an instance of {@link Procedure}.
     */
    <T> void asyncRequest(AsyncRead<T> request, Procedure<T> procedure);

    /**
     * Asynchronously determines the result of the given {@link MultiRead} and
     * registers the given {@link MultiRead} as a dependency of the ongoing
     * request if any. This is usually invoked for the side-effects of the given
     * {@link MultiRead} but can also be used for e.g. dependency registering or
     * cache priming. All exceptions from request processing will be lost.
     * 
     * @param request an instance of {@link MultiRead}.
     */
    <T> void asyncRequest(MultiRead<T> request);

    /**
     * Asynchronously registers the given {@link AsyncMultiListener} (as
     * identified by {@link Object#equals(Object)}) to track changes in the
     * result determined from the given {@link MultiRead}. Registers the given
     * {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given
     * {@link AsyncMultiListener}.
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link AsyncMultiListener}.
     */
    //<T> void asyncRequest(MultiRead<T> request, AsyncMultiListener<T> procedure);

    /**
     * Asynchronously registers the given {@link SyncMultiListener} (as
     * identified by {@link Object#equals(Object)}) to track changes in the
     * result determined from the given {@link MultiRead}. Registers the given
     * {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given
     * {@link SyncMultiListener}
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link SyncMultiListener}.
     */
    <T> void asyncRequest(MultiRead<T> request, SyncMultiListener<T> procedure);

    /**
     * Asynchronously registers the given {@link MultiListener} (as identified
     * by {@link Object#equals(Object)}) to track changes in the result
     * determined from the given {@link MultiRead}. Registers the given
     * {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given {@link MultiListener}
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link MultiListener}.
     */
    <T> void asyncRequest(MultiRead<T> request, MultiListener<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given
     * {@link MultiRead} to the given {@link AsyncMultiProcedure}. Registers the
     * given {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given
     * {@link AsyncMultiProcedure}
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link AsyncMultiProcedure}.
     */
    //<T> void asyncRequest(MultiRead<T> request, AsyncMultiProcedure<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given
     * {@link MultiRead} to the given {@link SyncMultiProcedure}. Registers the
     * given {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given
     * {@link SyncMultiProcedure}
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link SyncMultiProcedure}.
     */
    <T> void asyncRequest(MultiRead<T> request, SyncMultiProcedure<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given
     * {@link MultiRead} to the given {@link MultiProcedure}. Registers the
     * given {@link MultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link MultiRead} will be transferred to the given {@link MultiProcedure}
     * 
     * @param request an instance of {@link MultiRead}.
     * @param procedure an instance of {@link MultiProcedure}.
     */
    <T> void asyncRequest(MultiRead<T> request, MultiProcedure<T> procedure);

    /**
     * Asynchronously determines the result of the given {@link AsyncMultiRead}
     * and registers the given {@link AsyncMultiRead} as a dependency of the
     * ongoing request if any. This is usually invoked for the side-effects of
     * the given {@link AsyncMultiRead} but can also be used for e.g. dependency
     * registering or cache priming. All exceptions from request processing will
     * be lost.
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     */
    <T> void asyncRequest(AsyncMultiRead<T> request);

    /**
     * Asynchronously registers the given {@link AsyncMultiListener} (as
     * identified by {@link Object#equals(Object)}) to track changes in the
     * result determined from the given {@link AsyncMultiRead}. Registers the
     * given {@link AsyncMultiRead} as a dependency of the ongoing request if
     * any. Exceptions due to the determination of the result of the given
     * {@link AsyncMultiRead} will be transferred to the given
     * {@link AsyncMultiListener}.
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link AsyncMultiListener}.
     */
    <T> void asyncRequest(AsyncMultiRead<T> request, AsyncMultiListener<T> procedure);

    /**
     * Asynchronously registers the given {@link SyncMultiListener} (as
     * identified by {@link Object#equals(Object)}) to track changes in the
     * result determined from the given {@link AsyncMultiRead}. Registers the
     * given {@link AsyncMultiRead} as a dependency of the ongoing request if
     * any. Exceptions due to the determination of the result of the given
     * {@link AsyncMultiRead} will be transferred to the given
     * {@link SyncMultiListener}
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link SyncMultiListener}.
     */
    <T> void asyncRequest(AsyncMultiRead<T> request, SyncMultiListener<T> procedure);

    /**
     * Asynchronously registers the given {@link MultiListener} (as identified
     * by {@link Object#equals(Object)}) to track changes in the result
     * determined from the given {@link AsyncMultiRead}. Registers the given
     * {@link AsyncMultiRead} as a dependency of the ongoing request if any.
     * Exceptions due to the determination of the result of the given
     * {@link AsyncMultiRead} will be transferred to the given
     * {@link MultiListener}
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link MultiListener}.
     */
    <T> void asyncRequest(AsyncMultiRead<T> request, MultiListener<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given
     * {@link AsyncMultiRead} to the given {@link AsyncMultiProcedure}.
     * Registers the given {@link AsyncMultiRead} as a dependency of the ongoing
     * request if any. Exceptions due to the determination of the result of the
     * given {@link AsyncMultiRead} will be transferred to the given
     * {@link AsyncMultiProcedure}
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link AsyncMultiProcedure}.
     */
    <T> void asyncRequest(AsyncMultiRead<T> request, AsyncMultiProcedure<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given
     * {@link AsyncMultiRead} to the given {@link SyncMultiProcedure}. Registers
     * the given {@link AsyncMultiRead} as a dependency of the ongoing request
     * if any. Exceptions due to the determination of the result of the given
     * {@link AsyncMultiRead} will be transferred to the given
     * {@link SyncMultiProcedure}
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link SyncMultiProcedure}.
     */
    <T> void asyncRequest(AsyncMultiRead<T> request, SyncMultiProcedure<T> procedure);

    /**
     * Asynchronously supplies the result determined from the given
     * {@link AsyncMultiRead} to the given {@link MultiProcedure}. Registers the
     * given {@link AsyncMultiRead} as a dependency of the ongoing request if
     * any. Exceptions due to the determination of the result of the given
     * {@link AsyncMultiRead} will be transferred to the given
     * {@link MultiProcedure}
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     * @param procedure an instance of {@link MultiProcedure}.
     */
    <T> void asyncRequest(AsyncMultiRead<T> request, MultiProcedure<T> procedure);

    /**
     * Asynchronously determines the result of the given {@link AsyncMultiRead}
     * and registers the given {@link AsyncMultiRead} as a dependency of the
     * ongoing request if any. This is usually invoked for the side-effects of
     * the given {@link AsyncMultiRead} but can also be used for e.g. dependency
     * registering or cache priming. All exceptions from request processing will
     * be lost.
     * 
     * @param request an instance of {@link AsyncMultiRead}.
     */
    <T> void asyncRequest(ExternalRead<T> request);
    <T> void asyncRequest(ExternalRead<T> request, Listener<T> procedure);
    <T> void asyncRequest(ExternalRead<T> request, Procedure<T> procedure);
    
    
}
