/*******************************************************************************
 * Copyright (c) 2007, 2023 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db;

import java.util.Map;
import java.util.Set;

import org.simantics.databoard.binding.Binding;
import org.simantics.db.procedure.AsyncListener;
import org.simantics.db.procedure.AsyncMultiListener;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.AsyncSetListener;
import org.simantics.db.procedure.Listener;
import org.simantics.db.procedure.MultiListener;
import org.simantics.db.procedure.MultiProcedure;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.procedure.SetListener;
import org.simantics.db.procedure.SyncListener;
import org.simantics.db.procedure.SyncMultiListener;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.procedure.SyncProcedure;
import org.simantics.db.procedure.SyncSetListener;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.Read;

/**
 * 
 * For accessing and manipulating the graph data in asynchronous manner. A client
 * receives AsyncReadGraph instances for performing requests in {@link Read} and
 * {@link MultiRead} as well as for reacting to request results using
 * {@link AsyncProcedure}, {@link AsyncListener}, {@link AsyncMultiProcedure},
 * {@link AsyncMultiListener} and {@link AsyncSetListener}
 * <p>
 * AsyncReadGraph instances are stateful and may not be stored and used outside the
 * supplier method activation. The methods receiving AsyncReadGraph should not block for long
 * periods of time since that may delay the processing of other requests. For a
 * synchronous counterpart with the same functionality as AsyncReadGraph see
 * {@link ReadGraph}
 * 
 * @version 0.7
 * @author Antti Villberg
 * @see RequestProcessor
 * @see AsyncRequestProcessor
 * @see ReadGraph
 * @see Resource
 * @see Statement
 */
public interface AsyncReadGraph extends ReadGraph, AsyncRequestProcessor {

    /**
     * @see ReadGraph#getChildren(Resource)
     * @since 1.58.0
     */
    void forChildMap(Resource resource, AsyncProcedure<Map<String,Resource>> procedure);

    /**
     * @see ReadGraph#getURI(Resource)
     */
    void forURI(Resource resource, AsyncListener<String> procedure);
    /**
     * @see ReadGraph#getURI(Resource)
     */
    void forURI(Resource resource, SyncListener<String> procedure);
    /**
     * @see ReadGraph#getURI(Resource)
     */
    void forURI(Resource resource, Listener<String> procedure);
    /**
     * @see ReadGraph#getURI(Resource)
     */
    void forURI(Resource resource, AsyncProcedure<String> procedure);
    /**
     * @see ReadGraph#getURI(Resource)
     */
    void forURI(Resource resource, SyncProcedure<String> procedure);
    /**
     * @see ReadGraph#getURI(Resource)
     */
    void forURI(Resource resource, Procedure<String> procedure);

    /**
     * @see ReadGraph#getResourceByURI(String)
     */
    void forResource(String id, AsyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getResourceByURI(String)
     */
    void forResource(String id, SyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getResourceByURI(String)
     */
    void forResource(String id, Listener<Resource> procedure);
    /**
     * @see ReadGraph#getResourceByURI(String)
     */
    void forResource(String id, AsyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getResourceByURI(String)
     */
    void forResource(String id, SyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getResourceByURI(String)
     */
    void forResource(String id, Procedure<Resource> procedure);

    /**
     * @see ReadGraph#getBuiltin(String)
     */
    void forBuiltin(String id, AsyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getBuiltin(String)
     */
    void forBuiltin(String id, SyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getBuiltin(String)
     */
    void forBuiltin(String id, Listener<Resource> procedure);
    /**
     * @see ReadGraph#getBuiltin(String)
     */
    void forBuiltin(String id, AsyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getBuiltin(String)
     */
    void forBuiltin(String id, SyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getBuiltin(String)
     */
    void forBuiltin(String id, Procedure<Resource> procedure);

    /**
     * @see ReadGraph#getStatements(Resource, Resource)
     */
    void forEachStatement(Resource subject, Resource relation, AsyncMultiProcedure<Statement> procedure);
    /**
     * @see ReadGraph#getStatements(Resource, Resource)
     */
    void forEachStatement(Resource subject, Resource relation, SyncMultiProcedure<Statement> procedure);
    /**
     * @see ReadGraph#getStatements(Resource, Resource)
     */
    void forEachStatement(Resource subject, Resource relation, MultiProcedure<Statement> procedure);

    /**
     * @see ReadGraph#getStatements(Resource, Resource)
     */
    void forStatementSet(Resource subject, Resource relation, AsyncSetListener<Statement> procedure);
    /**
     * @see ReadGraph#getStatements(Resource, Resource)
     */
    void forStatementSet(Resource subject, Resource relation, SyncSetListener<Statement> procedure);
    /**
     * @see ReadGraph#getStatements(Resource, Resource)
     */
    void forStatementSet(Resource subject, Resource relation, SetListener<Statement> procedure);

    /**
     * @see ReadGraph#getAssertedStatements(Resource, Resource)
     */
    void forEachAssertedStatement(Resource subject, Resource relation, AsyncMultiProcedure<Statement> procedure);
    /**
     * @see ReadGraph#getAssertedStatements(Resource, Resource)
     */
    void forEachAssertedStatement(Resource subject, Resource relation, SyncMultiProcedure<Statement> procedure);
    /**
     * @see ReadGraph#getAssertedStatements(Resource, Resource)
     */
    void forEachAssertedStatement(Resource subject, Resource relation, MultiProcedure<Statement> procedure);

    /**
     * @see ReadGraph#getAssertedStatements(Resource, Resource)
     */
    void forAssertedStatementSet(Resource subject, Resource relation, AsyncSetListener<Statement> procedure);
    /**
     * @see ReadGraph#getAssertedStatements(Resource, Resource)
     */
    void forAssertedStatementSet(Resource subject, Resource relation, SyncSetListener<Statement> procedure);
    /**
     * @see ReadGraph#getAssertedStatements(Resource, Resource)
     */
    void forAssertedStatementSet(Resource subject, Resource relation, SetListener<Statement> procedure);

    /**
     * @see ReadGraph#getPredicates(Resource)
     */
    void forEachPredicate(Resource subject, AsyncMultiProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getPredicates(Resource)
     */
    void forEachPredicate(Resource subject, SyncMultiProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getPredicates(Resource)
     */
    void forEachPredicate(Resource subject, MultiProcedure<Resource> procedure);

    /**
     * @see ReadGraph#getPredicates(Resource)
     */
    void forPredicateSet(Resource subject, AsyncSetListener<Resource> procedure);
    /**
     * @see ReadGraph#getPredicates(Resource)
     */
    void forPredicateSet(Resource subject, SyncSetListener<Resource> procedure);
    /**
     * @see ReadGraph#getPredicates(Resource)
     */
    void forPredicateSet(Resource subject, SetListener<Resource> procedure);

    /**
     * @see ReadGraph#getPrincipalTypes(Resource)
     */
    void forEachPrincipalType(Resource subject, AsyncMultiProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getPrincipalTypes(Resource)
     */
    void forEachPrincipalType(Resource subject, SyncMultiProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getPrincipalTypes(Resource)
     */
    void forEachPrincipalType(Resource subject, MultiProcedure<Resource> procedure);

    /**
     * @see ReadGraph#getPrincipalTypes(Resource)
     */
    void forPrincipalTypeSet(Resource subject, AsyncSetListener<Resource> procedure);
    /**
     * @see ReadGraph#getPrincipalTypes(Resource)
     */
    void forPrincipalTypeSet(Resource subject, SyncSetListener<Resource> procedure);
    /**
     * @see ReadGraph#getPrincipalTypes(Resource)
     */
    void forPrincipalTypeSet(Resource subject, SetListener<Resource> procedure);

    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forTypes(Resource subject, AsyncListener<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forTypes(Resource subject, SyncListener<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forTypes(Resource subject, Listener<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forTypes(Resource subject, AsyncProcedure<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forTypes(Resource subject, SyncProcedure<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forTypes(Resource subject, Procedure<Set<Resource>> procedure);

    /**
     * @see ReadGraph#getSupertypes(Resource)
     */
    void forSupertypes(Resource subject, AsyncListener<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getSupertypes(Resource)
     */
    void forSupertypes(Resource subject, SyncListener<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getSupertypes(Resource)
     */
    void forSupertypes(Resource subject, Listener<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getSupertypes(Resource)
     */
    void forSupertypes(Resource subject, AsyncProcedure<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getSupertypes(Resource)
     */
    void forSupertypes(Resource subject, SyncProcedure<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getSupertypes(Resource)
     */
    void forSupertypes(Resource subject, Procedure<Set<Resource>> procedure);

    /**
     * @see ReadGraph#getSuperrelations(Resource)
     */
    void forDirectSuperrelations(Resource subject, AsyncMultiProcedure<Resource> procedure);

    /**
     * @see ReadGraph#getSuperrelations(Resource)
     */
    void forPossibleSuperrelation(Resource subject, AsyncProcedure<Resource> procedure);

    /**
     * @see ReadGraph#getSuperrelations(Resource)
     */
    void forSuperrelations(Resource subject, AsyncListener<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getSuperrelations(Resource)
     */
    void forSuperrelations(Resource subject, SyncListener<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getSuperrelations(Resource)
     */
    void forSuperrelations(Resource subject, Listener<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getSuperrelations(Resource)
     */
    void forSuperrelations(Resource subject, AsyncProcedure<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getSuperrelations(Resource)
     */
    void forSuperrelations(Resource subject, SyncProcedure<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getSuperrelations(Resource)
     */
    void forSuperrelations(Resource subject, Procedure<Set<Resource>> procedure);

    /**
     * 
     * If relation is functional this do one of the following
     * -call execute 0 or 1 times and then call finished
     * -call exception with ManyObjectsForFunctionalRelation
     * 
     * @see ReadGraph#getObjects(Resource, Resource)
     */
    void forEachObject(Resource subject, Resource relation, AsyncMultiProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getObjects(Resource, Resource)
     */
    void forEachObject(Resource subject, Resource relation, SyncMultiProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getObjects(Resource, Resource)
     */
    void forEachObject(Resource subject, Resource relation, MultiProcedure<Resource> procedure);

    /**
     * @see ReadGraph#getObjects(Resource, Resource)
     */
//    void forEachDirectObject(Resource subject, Resource relation, AsyncMultiProcedure<Resource> procedure);
//    /**
//     * @see ReadGraph#getObjects(Resource, Resource)
//     */
//    void forEachDirectObject(Resource subject, Resource relation, SyncMultiProcedure<Resource> procedure);
//    /**
//     * @see ReadGraph#getObjects(Resource, Resource)
//     */
//    void forEachDirectObject(Resource subject, Resource relation, MultiProcedure<Resource> procedure);

    /**
     * @see ReadGraph#getObjects(Resource, Resource)
     */
    void forEachDirectPredicate(Resource subject, AsyncProcedure<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getObjects(Resource, Resource)
     */
    void forEachDirectPredicate(Resource subject, SyncProcedure<Set<Resource>> procedure);
    /**
     * @see ReadGraph#getObjects(Resource, Resource)
     */
    void forEachDirectPredicate(Resource subject, Procedure<Set<Resource>> procedure);

    /**
     * @see ReadGraph#getObjects(Resource, Resource)
     */
    void forObjectSet(Resource subject, Resource relation, AsyncSetListener<Resource> procedure);
    /**
     * @see ReadGraph#getObjects(Resource, Resource)
     */
    void forObjectSet(Resource subject, Resource relation, SyncSetListener<Resource> procedure);
    /**
     * @see ReadGraph#getObjects(Resource, Resource)
     */
    void forObjectSet(Resource subject, Resource relation, SetListener<Resource> procedure);

    /**
     * @see ReadGraph#getAssertedObjects(Resource, Resource)
     */
    void forEachAssertedObject(Resource subject, Resource relation, AsyncMultiProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getAssertedObjects(Resource, Resource)
     */
    void forEachAssertedObject(Resource subject, Resource relation, SyncMultiProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getAssertedObjects(Resource, Resource)
     */
    void forEachAssertedObject(Resource subject, Resource relation, MultiProcedure<Resource> procedure);

    /**
     * @see ReadGraph#getAssertedObjects(Resource, Resource)
     */
    void forAssertedObjectSet(Resource subject, Resource relation, AsyncSetListener<Resource> procedure);
    /**
     * @see ReadGraph#getAssertedObjects(Resource, Resource)
     */
    void forAssertedObjectSet(Resource subject, Resource relation, SyncSetListener<Resource> procedure);
    /**
     * @see ReadGraph#getAssertedObjects(Resource, Resource)
     */
    void forAssertedObjectSet(Resource subject, Resource relation, SetListener<Resource> procedure);

    /**
     * @see ReadGraph#getInverse(Resource)
     */
    void forInverse(Resource relation, AsyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getInverse(Resource)
     */
    void forInverse(Resource relation, SyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getInverse(Resource)
     */
    void forInverse(Resource relation, Listener<Resource> procedure);
    /**
     * @see ReadGraph#getInverse(Resource)
     */
    void forInverse(Resource relation, AsyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getInverse(Resource)
     */
    void forInverse(Resource relation, SyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getInverse(Resource)
     */
    void forInverse(Resource relation, Procedure<Resource> procedure);

    /**
     * @see ReadGraph#getSingleObject(Resource, Resource)
     */
    void forSingleObject(Resource subject, Resource relation, AsyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getSingleObject(Resource, Resource)
     */
    void forSingleObject(Resource subject, Resource relation, SyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getSingleObject(Resource, Resource)
     */
    void forSingleObject(Resource subject, Resource relation, Listener<Resource> procedure);
    /**
     * @see ReadGraph#getSingleObject(Resource, Resource)
     */
    void forSingleObject(Resource subject, Resource relation, AsyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getSingleObject(Resource, Resource)
     */
    void forSingleObject(Resource subject, Resource relation, SyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getSingleObject(Resource, Resource)
     */
    void forSingleObject(Resource subject, Resource relation, Procedure<Resource> procedure);

    /**
     * @see ReadGraph#getSingleStatement(Resource, Resource)
     */
    void forSingleStatement(Resource subject, Resource relation, AsyncListener<Statement> procedure);
    /**
     * @see ReadGraph#getSingleStatement(Resource, Resource)
     */
    void forSingleStatement(Resource subject, Resource relation, SyncListener<Statement> procedure);
    /**
     * @see ReadGraph#getSingleStatement(Resource, Resource)
     */
    void forSingleStatement(Resource subject, Resource relation, Listener<Statement> procedure);
    /**
     * @see ReadGraph#getSingleStatement(Resource, Resource)
     */
    void forSingleStatement(Resource subject, Resource relation, AsyncProcedure<Statement> procedure);
    /**
     * @see ReadGraph#getSingleStatement(Resource, Resource)
     */
    void forSingleStatement(Resource subject, Resource relation, SyncProcedure<Statement> procedure);
    /**
     * @see ReadGraph#getSingleStatement(Resource, Resource)
     */
    void forSingleStatement(Resource subject, Resource relation, Procedure<Statement> procedure);

    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, AsyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, SyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, Listener<Resource> procedure);
    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, AsyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, SyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, Procedure<Resource> procedure);

    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, Resource baseType, AsyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, Resource baseType, SyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, Resource baseType, Listener<Resource> procedure);
    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, Resource baseType, AsyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, Resource baseType, SyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getSingleType(Resource, Resource)
     */
    void forSingleType(Resource subject, Resource baseType, Procedure<Resource> procedure);

    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, SyncListener<T> procedure);
    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, Listener<T> procedure);
    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, Procedure<T> procedure);

    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, Binding binding, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, Binding binding, SyncListener<T> procedure);
    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, Binding binding, Listener<T> procedure);
    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, Binding binding, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, Binding binding, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getValue(Resource)
     */
    <T> void forValue(Resource subject, Binding binding, Procedure<T> procedure);

    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, SyncListener<T> procedure);
    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, Listener<T> procedure);
    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, Procedure<T> procedure);

    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource, Binding)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, Binding binding, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource, Binding)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, Binding binding, SyncListener<T> procedure);
    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource, Binding)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, Binding binding, Listener<T> procedure);
    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource, Binding)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, Binding binding, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource, Binding)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, Binding binding, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getRelatedValue(Resource, Resource, Binding)
     */
    <T> void forRelatedValue(Resource subject, Resource relation, Binding binding, Procedure<T> procedure);

    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forAdapted(Resource resource, Class<T> clazz, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forAdapted(Resource resource, Class<T> clazz, SyncListener<T> procedure);
    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forAdapted(Resource resource, Class<T> clazz, Listener<T> procedure);
    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forAdapted(Resource resource, Class<T> clazz, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forAdapted(Resource resource, Class<T> clazz, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forAdapted(Resource resource, Class<T> clazz, Procedure<T> procedure);

    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forUniqueAdapted(Resource resource, Class<T> clazz, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forUniqueAdapted(Resource resource, Class<T> clazz, SyncListener<T> procedure);
    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forUniqueAdapted(Resource resource, Class<T> clazz, Listener<T> procedure);
    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forUniqueAdapted(Resource resource, Class<T> clazz, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forUniqueAdapted(Resource resource, Class<T> clazz, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#adapt(Resource, Class)
     */
    <T> void forUniqueAdapted(Resource resource, Class<T> clazz, Procedure<T> procedure);

    /**
     * @see ReadGraph#getPossibleInverse(Resource)
     */
    void forPossibleInverse(Resource relation, AsyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleInverse(Resource)
     */
    void forPossibleInverse(Resource relation, SyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleInverse(Resource)
     */
    void forPossibleInverse(Resource relation, Listener<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleInverse(Resource)
     */
    void forPossibleInverse(Resource relation, AsyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleInverse(Resource)
     */
    void forPossibleInverse(Resource relation, SyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleInverse(Resource)
     */
    void forPossibleInverse(Resource relation, Procedure<Resource> procedure);

    /**
     * @see ReadGraph#getPossibleObject(Resource, Resource)
     */
    void forPossibleObject(Resource subject, Resource relation, AsyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleObject(Resource, Resource)
     */
    void forPossibleObject(Resource subject, Resource relation, SyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleObject(Resource, Resource)
     */
    void forPossibleObject(Resource subject, Resource relation, Listener<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleObject(Resource, Resource)
     */
    void forPossibleObject(Resource subject, Resource relation, AsyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleObject(Resource, Resource)
     */
    void forPossibleObject(Resource subject, Resource relation, SyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleObject(Resource, Resource)
     */
    void forPossibleObject(Resource subject, Resource relation, Procedure<Resource> procedure);

    /**
     * @see ReadGraph#getPossibleStatement(Resource, Resource)
     */
    void forPossibleStatement(Resource subject, Resource relation, AsyncListener<Statement> procedure);
    /**
     * @see ReadGraph#getPossibleStatement(Resource, Resource)
     */
    void forPossibleStatement(Resource subject, Resource relation, SyncListener<Statement> procedure);
    /**
     * @see ReadGraph#getPossibleStatement(Resource, Resource)
     */
    void forPossibleStatement(Resource subject, Resource relation, Listener<Statement> procedure);
    /**
     * @see ReadGraph#getPossibleStatement(Resource, Resource)
     */
    void forPossibleStatement(Resource subject, Resource relation, AsyncProcedure<Statement> procedure);
    /**
     * @see ReadGraph#getPossibleStatement(Resource, Resource)
     */
    void forPossibleStatement(Resource subject, Resource relation, SyncProcedure<Statement> procedure);
    /**
     * @see ReadGraph#getPossibleStatement(Resource, Resource)
     */
    void forPossibleStatement(Resource subject, Resource relation, Procedure<Statement> procedure);

    /**
     * @see ReadGraph#getPossibleType(Resource, Resource)
     */
    void forPossibleType(Resource subject, Resource baseType, AsyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleType(Resource, Resource)
     */
    void forPossibleType(Resource subject, Resource baseType, SyncListener<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleType(Resource, Resource)
     */
    void forPossibleType(Resource subject, Resource baseType, Listener<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleType(Resource, Resource)
     */
    void forPossibleType(Resource subject, Resource baseType, AsyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleType(Resource, Resource)
     */
    void forPossibleType(Resource subject, Resource baseType, SyncProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getPossibleType(Resource, Resource)
     */
    void forPossibleType(Resource subject, Resource baseType, Procedure<Resource> procedure);

    /**
     * @see ReadGraph#getPossibleValue(Resource)
     */
    <T> void forPossibleValue(Resource subject, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleValue(Resource)
     */
    <T> void forPossibleValue(Resource subject, SyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleValue(Resource)
     */
    <T> void forPossibleValue(Resource subject, Listener<T> procedure);
    /**
     * @see ReadGraph#getPossibleValue(Resource)
     */
    <T> void forPossibleValue(Resource subject, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleValue(Resource)
     */
    <T> void forPossibleValue(Resource subject, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleValue(Resource)
     */
    <T> void forPossibleValue(Resource subject, Procedure<T> procedure);

    /**
     * @see ReadGraph#getPossibleValue(Resource, Binding)
     */
    <T> void forPossibleValue(Resource subject, Binding binding, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleValue(Resource, Binding)
     */
    <T> void forPossibleValue(Resource subject, Binding binding, SyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleValue(Resource, Binding)
     */
    <T> void forPossibleValue(Resource subject, Binding binding, Listener<T> procedure);
    /**
     * @see ReadGraph#getPossibleValue(Resource, Binding)
     */
    <T> void forPossibleValue(Resource subject, Binding binding, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleValue(Resource, Binding)
     */
    <T> void forPossibleValue(Resource subject, Binding binding, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleValue(Resource, Binding)
     */
    <T> void forPossibleValue(Resource subject, Binding binding, Procedure<T> procedure);

    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, SyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, Listener<T> procedure);
    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, Procedure<T> procedure);

    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource, Binding)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, Binding binding, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource, Binding)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, Binding binding, SyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource, Binding)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, Binding binding, Listener<T> procedure);
    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource, Binding)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, Binding binding, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource, Binding)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, Binding binding, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleRelatedValue(Resource, Resource, Binding)
     */
    <T> void forPossibleRelatedValue(Resource subject, Resource relation, Binding binding, Procedure<T> procedure);

    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleAdapted(Resource resource, Class<T> clazz, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleAdapted(Resource resource, Class<T> clazz, SyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleAdapted(Resource resource, Class<T> clazz, Listener<T> procedure);
    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleAdapted(Resource resource, Class<T> clazz, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleAdapted(Resource resource, Class<T> clazz, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleAdapted(Resource resource, Class<T> clazz, Procedure<T> procedure);

    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz, AsyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz, SyncListener<T> procedure);
    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz, Listener<T> procedure);
    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz, AsyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz, SyncProcedure<T> procedure);
    /**
     * @see ReadGraph#getPossibleAdapter(Resource, Class)
     */
    <T> void forPossibleUniqueAdapted(Resource resource, Class<T> clazz, Procedure<T> procedure);

    /**
     * @see ReadGraph#isInstanceOf(Resource, Resource)
     */
    void forIsInstanceOf(Resource resource, Resource type, AsyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#isInstanceOf(Resource, Resource)
     */
    void forIsInstanceOf(Resource resource, Resource type, SyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#isInstanceOf(Resource, Resource)
     */
    void forIsInstanceOf(Resource resource, Resource type, Listener<Boolean> procedure);
    /**
     * @see ReadGraph#isInstanceOf(Resource, Resource)
     */
    void forIsInstanceOf(Resource resource, Resource type, AsyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#isInstanceOf(Resource, Resource)
     */
    void forIsInstanceOf(Resource resource, Resource type, SyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#isInstanceOf(Resource, Resource)
     */
    void forIsInstanceOf(Resource resource, Resource type, Procedure<Boolean> procedure);

    /**
     * @see ReadGraph#isInheritedFrom(Resource, Resource)
     */
    void forIsInheritedFrom(Resource resource, Resource type, AsyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#isInheritedFrom(Resource, Resource)
     */
    void forIsInheritedFrom(Resource resource, Resource type, SyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#isInheritedFrom(Resource, Resource)
     */
    void forIsInheritedFrom(Resource resource, Resource type, Listener<Boolean> procedure);
    /**
     * @see ReadGraph#isInheritedFrom(Resource, Resource)
     */
    void forIsInheritedFrom(Resource resource, Resource type, AsyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#isInheritedFrom(Resource, Resource)
     */
    void forIsInheritedFrom(Resource resource, Resource type, SyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#isInheritedFrom(Resource, Resource)
     */
    void forIsInheritedFrom(Resource resource, Resource type, Procedure<Boolean> procedure);

    /**
     * @see ReadGraph#isSubrelationOf(Resource, Resource)
     */
    void forIsSubrelationOf(Resource resource, Resource relation, AsyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#isSubrelationOf(Resource, Resource)
     */
    void forIsSubrelationOf(Resource resource, Resource relation, SyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#isSubrelationOf(Resource, Resource)
     */
    void forIsSubrelationOf(Resource resource, Resource relation, Listener<Boolean> procedure);
    /**
     * @see ReadGraph#isSubrelationOf(Resource, Resource)
     */
    void forIsSubrelationOf(Resource resource, Resource relation, AsyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#isSubrelationOf(Resource, Resource)
     */
    void forIsSubrelationOf(Resource resource, Resource relation, SyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#isSubrelationOf(Resource, Resource)
     */
    void forIsSubrelationOf(Resource resource, Resource relation, Procedure<Boolean> procedure);

    /**
     * @see ReadGraph#hasStatement(Resource)
     */
    void forHasStatement(Resource subject, AsyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource)
     */
    void forHasStatement(Resource subject, SyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource)
     */
    void forHasStatement(Resource subject, Listener<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource)
     */
    void forHasStatement(Resource subject, AsyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource)
     */
    void forHasStatement(Resource subject, SyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource)
     */
    void forHasStatement(Resource subject, Procedure<Boolean> procedure);

    /**
     * @see ReadGraph#hasStatement(Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, AsyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, SyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, Listener<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, AsyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, SyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, Procedure<Boolean> procedure);

    /**
     * @see ReadGraph#hasStatement(Resource, Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, Resource object, AsyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource, Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, Resource object, SyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource, Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, Resource object, Listener<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource, Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, Resource object, AsyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource, Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, Resource object, SyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#hasStatement(Resource, Resource, Resource)
     */
    void forHasStatement(Resource subject, Resource relation, Resource object, Procedure<Boolean> procedure);

    /**
     * @see ReadGraph#hasValue(Resource)
     */
    void forHasValue(Resource subject, AsyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#hasValue(Resource)
     */
    void forHasValue(Resource subject, SyncListener<Boolean> procedure);
    /**
     * @see ReadGraph#hasValue(Resource)
     */
    void forHasValue(Resource subject, Listener<Boolean> procedure);
    /**
     * @see ReadGraph#hasValue(Resource)
     */
    void forHasValue(Resource subject, AsyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#hasValue(Resource)
     */
    void forHasValue(Resource subject, SyncProcedure<Boolean> procedure);
    /**
     * @see ReadGraph#hasValue(Resource)
     */
    void forHasValue(Resource subject, Procedure<Boolean> procedure);

    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forOrderedSet(Resource subject, AsyncMultiListener<Resource> procedure);
    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forOrderedSet(Resource subject, SyncMultiListener<Resource> procedure);
    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forOrderedSet(Resource subject, MultiListener<Resource> procedure);
    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forOrderedSet(Resource subject, AsyncMultiProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forOrderedSet(Resource subject, SyncMultiProcedure<Resource> procedure);
    /**
     * @see ReadGraph#getTypes(Resource)
     */
    void forOrderedSet(Resource subject, MultiProcedure<Resource> procedure);

//    void inc();
//    void dec();
    
    boolean performPending();
    
}
