/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.procore.cluster;

import gnu.trove.map.hash.TObjectIntHashMap;

import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ClusterTraitsBase;
import org.simantics.db.service.ClusterUID;
import org.simantics.db.service.ResourceUID;

public class ClusterMap {
    private ForeignTable foreignTable;
    //private FlatTable flatTable;
    private TObjectIntHashMap<ClusterUID> hashMap;
    public ClusterMap(ForeignTable foreignTable, FlatTable flatTable) {
        this.foreignTable = foreignTable;
        //this.flatTable = flatTable;
        hashMap = null;
    }
    public void compact() {
        if(hashMap == null)
            return;
        hashMap.compact();
    }
    /*
     * 13 bytes for each element. Approx capacity = 3 * size
     */
    public int getUsedSpace() {
        if(hashMap == null)
            return 0;
        return 3 * 13 * hashMap.size();
    }
    public int getForeignResourceIndex(int reference)
    throws DatabaseException {
        if (ClusterTraits.isFlat(reference))
            throw new DatabaseException("Flat reference is not supported yet.");
        else {
            return ClusterTraits.getResourceIndexFromForeignReference(reference);
        }
    }
    public final ResourceUID getResourceUID(int reference) throws DatabaseException {
        if (ClusterTraits.isFlat(reference))
            throw new DatabaseException("Flat reference is not supported yet. reference=" + reference);
        else {
            int foreignIndex = ClusterTraits.getForeignIndexFromReference(reference);
            return foreignTable.getResourceUID(foreignIndex);
        }
    }
    public int getForeignReferenceOrCreateByResourceKey(int resourceKey, ClusterUID clusterUID)
    throws DatabaseException {
        int resourceIndex = ClusterTraits.getResourceIndexFromResourceKey(resourceKey);
        return getForeignReferenceOrCreateByResourceIndex(resourceIndex, clusterUID);
    }
    public int getForeignReferenceOrCreateByResourceIndex(int resourceIndex, ClusterUID clusterUID)
    throws DatabaseException {
        if (ClusterTraitsBase.isIllegalResourceIndex(resourceIndex))
            throw new DatabaseException("Illegal resource index=" + resourceIndex + ".");
        if (null == hashMap)
            hashMap = foreignTable.getHashMap();
        int foreignIndex = hashMap.get(clusterUID);
        if (0 == foreignIndex) {
            foreignIndex = foreignTable.createForeign(clusterUID.toRID(resourceIndex));
            hashMap.put(clusterUID, foreignIndex);
        }
        return ClusterTraits.createForeignReference(foreignIndex, resourceIndex); 
    }
    public int getForeignReferenceOrZero(int resourceIndex, ClusterUID clusterUID)
    throws DatabaseException {
        // This is guaranteed by implementation
//        if(clusterId == 1) return (resourceIndex<<16) + 0x80000001;
        //return ClusterTraits.createForeignReference(1, resourceIndex); 
//        new Exception("getForeignReferenceOrZero " + clusterId).printStackTrace();
        if (null == hashMap)
            hashMap = foreignTable.getHashMap();
        if (ClusterUID.Null.equals(clusterUID))
            throw new DatabaseException("Illegal cluster uid " + clusterUID);
        if (0 == resourceIndex)
            throw new DatabaseException("Illegal resource index " + resourceIndex);
        int foreignIndex = hashMap.get(clusterUID);
        if (0 == foreignIndex)
            return 0;
        //System.err.println("foreignIndex for " + clusterId + " = " + foreignIndex);
        return ClusterTraits.createForeignReference(foreignIndex, resourceIndex); 
    }
}
