/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package fi.vtt.simantics.procore.internal;

import java.util.Collection;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.graph.WriteGraphImpl;
import org.simantics.db.impl.query.QueryThread;
import org.simantics.db.procedure.AsyncContextMultiProcedure;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.Read;
import org.simantics.db.service.QueryControl;
import org.simantics.utils.DataContainer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueryControlImpl implements QueryControl {

	private static final Logger LOGGER = LoggerFactory.getLogger(QueryControlImpl.class);

	final private SessionImplSocket session;

	QueryControlImpl(SessionImplSocket session) {
		this.session = session;
	}

	@Override
	public int getAmountOfQueryThreads() {
		return session.getAmountOfQueryThreads();
	}

	@Override
	public int getGraphThread(AsyncReadGraph graph) {
		QueryThread thread = QueryThread.threadLocal.get();
		return thread.getIndex();
	}

	@Override
	public int flush() {
		final DataContainer<Integer> result = new DataContainer<Integer>(); 
		try {
			session.syncRequest(new WriteRequest() {

				@Override
				public void perform(WriteGraph graph) throws DatabaseException {
					session.queryProvider2.invalidateQueries((WriteGraphImpl)graph);
					result.set(0);
				}

			});
		} catch (DatabaseException e) {
			LOGGER.error("query flush failed", e);
		}
		return result.get();
	}

	@Override
	public int flush(ReadGraph graph) {
		return session.queryProvider2.clean();
	}

	@Override
	public int count() {
		return session.queryProvider2.querySize();
	}

	@Override
	public void gc(ReadGraph graph, int allowedTimeInMs) {
		// 20% young target
		session.queryProvider2.gc(20, allowedTimeInMs);
	}

	@Override
	public void gc(final Collection<ExternalRead<?>> requests) {
		try {
			session.syncRequest(new WriteRequest() {
				@Override
				public void perform(WriteGraph graph) throws DatabaseException {
					gc(graph, requests);
				}
			});
		} catch (DatabaseException e) {
			LOGGER.error("query gc failed", e);
		}
	}

	@Override
	public void gc(WriteGraph graph, final Collection<ExternalRead<?>> requests) {
		if (graph == null)
			throw new IllegalArgumentException("null WriteGraph");
		if (requests == null)
			throw new IllegalArgumentException("null requests");
		session.queryProvider2.clean(requests);
	}

	@Override
	public void invalidateQueries(WriteGraph graph) {
		session.queryProvider2.invalidateQueries((WriteGraphImpl)graph);
	}

	@Override
	public boolean scheduleByCluster(AsyncReadGraph graph, final Resource resource, final AsyncMultiProcedure<Resource> procedure) {
		final ReadGraphImpl impl = (ReadGraphImpl)graph;
		ResourceImpl res = (ResourceImpl)resource;
		int targetThread = ((res.id >>> 16) & session.queryProvider2.THREAD_MASK);
		if(0 == targetThread) return true;
		//System.err.println("scheduleByCluster[" + res.id + "|" + (res.id>>>16) + "] " + impl.callerThread + " -> " + targetThread);
		//		impl.state.barrier.inc();
		//		
		//		AsyncReadGraph targetGraph = impl.newAsync();
		procedure.execute(impl, resource);
		//		impl.state.barrier.dec();

		return false;

	}

	@Override
	public <C> boolean scheduleByCluster(AsyncReadGraph graph, final Resource resource, final C context, final AsyncContextMultiProcedure<C, Resource> procedure) {
		final ReadGraphImpl impl = (ReadGraphImpl)graph;
		ResourceImpl res = (ResourceImpl)resource;
		int targetThread = ((res.id >>> 16) & session.queryProvider2.THREAD_MASK);
		if(0 == targetThread) return true;
		//System.err.println("scheduleByCluster[" + res.id + "|" + (res.id>>>16) + "] " + impl.callerThread + " -> " + targetThread);
		//		impl.state.barrier.inc();
		//		
		//		AsyncReadGraph targetGraph = impl.newAsync();
		procedure.execute(impl, context, resource);
		//		impl.state.barrier.dec();

		return false;
	}

	@Override
	public void schedule(AsyncReadGraph graph, int targetThread, final ControlProcedure procedure) {
		final ReadGraphImpl impl = (ReadGraphImpl)graph;

		//		impl.state.barrier.inc();
		//
		//		AsyncReadGraph targetGraph = impl.newAsync();
		procedure.execute(impl);
		//		impl.state.barrier.dec();

	}

	@Override
	public ReadGraph getIndependentGraph(ReadGraph graph) {
		ReadGraphImpl impl = (ReadGraphImpl)graph;
		ReadGraphImpl result = impl.withParent(null, null, false);
		// Do not wait for the result graph
		impl.asyncBarrier.dec();
		return result;
	}

	@Override
	public <T> T syncRequestIndependent(ReadGraph graph, Read<T> request) throws DatabaseException {
		ReadGraphImpl independent = ((ReadGraphImpl)graph).withParent(null, null, false);
		try {
			return independent.syncRequest(request);
		} finally {
			independent.asyncBarrier.dec();
		}
	}

	@Override
	public boolean hasParentRequest(ReadGraph graph) {
		ReadGraphImpl impl = (ReadGraphImpl)graph;
		return impl.parent != null;
	}

}
