package fi.vtt.simantics.procore.internal;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.db.IO;
import org.simantics.db.exception.RuntimeDatabaseException;
import org.simantics.db.impl.ClusterTraitsBase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.set.hash.TLongHashSet;

public class ClusterSets {

    final static Logger LOGGER = LoggerFactory.getLogger(ClusterSets.class); 

	private IO write;
	//final private String databaseId;
	private HashMap<Long, Long> clusterSets; // Maps cluster set resource id to last user cluster id in the cluster set.
	private HashMap<Long, Long> reverseMap;
	private HashSet<Long> immutable;
	private int refCount; // Reference counter for user of this class.
	private boolean modified; // True if modified since last save.

	public void setWriteDirectory(IO write) {
		this.write = write;
	}

	public boolean tryLoad1(IO read) throws Exception {

		try {
			byte[] blobBytes = read.readBytes(0, (int)read.length());
			ByteArrayInputStream bais = new ByteArrayInputStream(blobBytes);
			PersistentDataBlob blob = (PersistentDataBlob) org.simantics.databoard.Files.readFile(bais, PersistentDataBlob.BINDING);
			bais.close();
			this.clusterSets = blob.clusterSets;
			this.reverseMap = blob.reverseMap;
			this.immutable = new HashSet<>();
			this.modified = false;
		} catch (IOException e1) {
			return false;
		}

		/*LOGGER.info("Loaded ClusterSets");
		LOGGER.info(" -forward");
		for(var e : clusterSets.entrySet())
			LOGGER.info(" -" + e.getKey() + " => " + e.getValue());
		LOGGER.info(" -reverse");
		for(var e : reverseMap.entrySet())
			LOGGER.info(" -" + e.getKey() + " => " + e.getValue());*/
		
		return true;

	}
	
	public boolean tryLoad2(IO read) throws Exception {

		try {
			byte[] blobBytes = read.readBytes(0, (int)read.length());
			ByteArrayInputStream bais = new ByteArrayInputStream(blobBytes);
			PersistentDataBlob2 blob = (PersistentDataBlob2) org.simantics.databoard.Files.readFile(bais, PersistentDataBlob2.BINDING);
			bais.close();
			this.clusterSets = blob.clusterSets;
			this.reverseMap = blob.reverseMap;
			this.immutable = blob.immutable;
			this.modified = false;
		} catch (IOException e1) {
			return false;
		}

		/*LOGGER.info("Loaded ClusterSets");
		LOGGER.info(" -forward");
		for(var e : clusterSets.entrySet())
			LOGGER.info(" -" + e.getKey() + " => " + e.getValue());
		LOGGER.info(" -reverse");
		for(var e : reverseMap.entrySet())
			LOGGER.info(" -" + e.getKey() + " => " + e.getValue());*/
		
		return true;

	}
	
	public void create(IO read) throws Exception {
		PersistentDataBlob2 blob = new PersistentDataBlob2();
		blob.clusterSets = new HashMap<>();
		blob.reverseMap = new HashMap<>();
		blob.immutable = new HashSet<>();
		doSave2(blob);
		this.clusterSets = blob.clusterSets;
		this.reverseMap = blob.reverseMap;
		this.immutable = blob.immutable;
	}

	public ClusterSets(IO read, IO write, String databaseId) {

		try {

			//this.databaseId = databaseId;
			this.write = write;

			if(!tryLoad1(read)) {
				if(!tryLoad2(read)) {
					create(read);
				}
			}
			
		} catch (Exception e) {
			
			throw new RuntimeDatabaseException("Failed to create ClusterSets.", e);
			
		}

	}
	public synchronized int inc() {
		return ++refCount;
	}
	public synchronized int dec() {
		return --refCount;
	}
	// Save is thread safe.
	public void dispose() {
		try {
			// We still save changes even if transaction is not finished.
			// This is because we have no cancel mechanism for cluster (sets).
			save();
		} catch (IOException e) {
			e.printStackTrace();
			throw new RuntimeDatabaseException("Failed to save ClusterSets.");
		} 
	}
	// clusterSets is not thread safe.
	public synchronized boolean containsKey(long resourceId) {
		return clusterSets.containsKey(resourceId);
	}
	public synchronized Long get(Long resourceId) {
		return clusterSets.get(resourceId);
	}
	public synchronized void setActiveCluster(long clusterSetResourceId, long clusterId) {
		clusterSets.put(clusterSetResourceId, clusterId);
		reverseMap.put(clusterId, clusterSetResourceId);
		modified = true;
	}
	public synchronized void setImmutability(long clusterSetResourceId, boolean value) {
		if(value) {
			modified |= immutable.add(clusterSetResourceId);
		} else {
			modified |= immutable.remove(clusterSetResourceId);
		}
	}
	public synchronized Long getClusterSet(Long clusterId) {
		return reverseMap.get(clusterId);
	}
	public void clear() {
		
		for(Long key : clusterSets.keySet())
			clusterSets.put(key, -1L);
		
		touch();
		
	}
	
	public synchronized void touch() {
		modified = true;
	}
	
	private synchronized PersistentDataBlob2 doSave2(PersistentDataBlob2 blob)  throws IOException {
		byte[] bytes = blob.bytes();
		write.saveBytes(bytes, bytes.length, true);
		modified = false;
		return blob;
	}

	public synchronized void save() throws IOException {

		if (!modified)
			return;
		
		PersistentDataBlob2 blob = new PersistentDataBlob2();
		blob.clusterSets = clusterSets;
		blob.reverseMap = reverseMap;
		blob.immutable = immutable;
		doSave2(blob);

	}
	
	static public class PersistentDataBlob {
		final public static Binding BINDING = Bindings.getBindingUnchecked(PersistentDataBlob.class);
		public HashMap<Long, Long> clusterSets;
		public HashMap<Long, Long> reverseMap;
		public byte[] bytes() throws IOException {
			return Serialization.toByteArray(4096, BINDING, this);
		}
	}


	static public class PersistentDataBlob2 {
		final public static Binding BINDING = Bindings.getBindingUnchecked(PersistentDataBlob2.class);
		public HashMap<Long, Long> clusterSets;
		public HashMap<Long, Long> reverseMap;
		public HashSet<Long> immutable;
		public byte[] bytes() throws IOException {
			return Serialization.toByteArray(4096, BINDING, this);
		}
	}

	public void validate(long[] ids) {

		// clusterSets: cluster set persistent resource id (long) -> cluster id (long)
		// reverseMap: cluster id (long) -> cluster set persistent resource id (long)

		TLongHashSet idSet = new TLongHashSet(ids);

		ArrayList<Long> clusterSetResourceIds = new ArrayList<>(clusterSets.keySet());
		for(long setResourceId : clusterSetResourceIds) {
			long setClusterId = ClusterTraitsBase.getClusterIdFromResourceId(setResourceId);
			if(!idSet.contains(setClusterId)) {
				clusterSets.remove(setResourceId);
				LOGGER.info("Remove cluster set mapping: cluster set has been removed " + setResourceId);
			} else {
				long clusterId = clusterSets.get(setResourceId);
				if(clusterId == -1)
					continue;
				if(!idSet.contains(clusterId)) {
					clusterSets.put(setResourceId, -1L);
					LOGGER.info("Reset cluster set mapping: active cluster set cluster has been removed " + setResourceId + " " + clusterId);
				}
			}
		}
	
		ArrayList<Long> setClusterIds = new ArrayList<>(reverseMap.keySet());
		for(Long clusterId : setClusterIds) {
			if(clusterId == -1)
				continue;
			if(!idSet.contains(clusterId)) {
				reverseMap.remove(clusterId);
				LOGGER.info("Remove cluster set reverse mapping: cluster has been removed " + clusterId);
			} else {
				Long setResourceId = reverseMap.get(clusterId);
				long setClusterId = ClusterTraitsBase.getClusterIdFromResourceId(setResourceId);
				if(!idSet.contains(setClusterId)) {
					reverseMap.remove(clusterId);
					LOGGER.info("Remove cluster set reverse mapping: cluster set cluster has been removed " + setClusterId);
				}
			}
		}

	}

}
