/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.procore.cluster;

import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ClusterI;
import org.simantics.db.impl.ClusterTraitsBase;


public final class ClusterTraits extends ClusterTraitsBase {
    private static final int FOREIGN_INDEX_BITS = 16;
    private static final int FOREIGN_INDEX_MAX = (1<<FOREIGN_INDEX_BITS)-1;
    private static final int CLUSTER_INDEX_BITS = 16;
    private static final int CLUSTER_INDEX_MAX =  (1<<CLUSTER_INDEX_BITS)-1;
    private static final int TYPE_BITS = 2;
    private static final int NOTYPE_BITS = 32 - TYPE_BITS;
//    private static final int CLUSTER_RESERVED_BITS = 15;
//    private static final long CLUSTER_RESERVED_BITS2ZERO = ((long)1 << CLUSTER_RESERVED_BITS) - 1; 
    public static final int createForeignReference(int foreignIndex, int resourceIndex)
    throws DatabaseException {
        if (foreignIndex < 1 || foreignIndex > FOREIGN_INDEX_MAX)
            throw new DatabaseException("Illegal foreign index " + foreignIndex);
        if (isIllegalResourceIndex(resourceIndex))
            throw new DatabaseException("Illegal resource index " + resourceIndex);
        return (resourceIndex<<FOREIGN_INDEX_BITS | foreignIndex)| 0x80000000;
    }
    public static int referenceGetType(int ref) {
        return ref >>> NOTYPE_BITS;
    }
    public static final boolean isFlat(int reference) {
        return reference < 0 && (reference & 0x40000000) != 0;
    }
    public static final boolean isLocal(int reference) {
        return (reference & 0xC0000000) == 0;
    }
    public static final int getForeignIndexFromReference(int reference)
    throws DatabaseException {
        int foreignIndex = reference & 0x0000FFFF;
        if (0 == foreignIndex)
            throw new DatabaseException("Illegal foreign index " + foreignIndex);
        return foreignIndex;
    }
    public static final int getResourceIndexFromForeignReference(int reference) 
    throws DatabaseException {
        int resourceIndex = (reference & 0x7FFFFFFF) >> FOREIGN_INDEX_BITS;
        if (isIllegalResourceIndex(resourceIndex))
            throw new DatabaseException("Illegal foreign reference " + reference);
        return resourceIndex;
    }
    public static final boolean statementIndexIsDirect(int index) {
        return (index & 0xC0000000) != 0x40000000;
    }
    public static final int statementIndexMake(int index)
    throws DatabaseException {
        assert(index > 0);
        if ((index & 0xC0000000) != 0)
            throw new DatabaseException("Illegal statement index " + index);
        return index | 0x40000000;
    }
    public static final int statementIndexGet(int index)
    throws DatabaseException {
        if ((index & 0xC0000000) == 0x40000000)
            return index & 0x3FFFFFFF;
        else
            return index;
    }
    public static boolean completeReferenceIsLocal(int completeRef) {
        boolean complete = (completeRef & 0xC0000000) != 0;
        return complete && ((completeRef & 0x0000FFFF) == 0);
    }
    public static boolean completeReferenceIsMultiple(int completeRef) {
        boolean complete = (completeRef & 0xC0000000) != 0;
        return !complete && (completeRef != 0);
    }
    public static int completeReferenceGetResourceIndex(int completeRef) {
        int low = (completeRef & 0x3FFFFFFF) >>> CLUSTER_INDEX_BITS;
        return low & 0x00003FFFF;
    }
    public static int completeReferenceGetForeignIndex(int completeRef) {
        return completeRef & 0x0000FFFF;
    }
    public static ClusterI.CompleteTypeEnum completeReferenceGetType(int completeRef) {
        return ClusterI.CompleteTypeEnum.make(completeRef >>> NOTYPE_BITS);
    }
    public static int completeReferenceMake(int type, int resourceIndex, int clusterIndex) {
        assert(type < (1<<TYPE_BITS));
        assert(!isIllegalResourceIndex(resourceIndex));
        assert(clusterIndex <= CLUSTER_INDEX_MAX);
        int ref = (type << NOTYPE_BITS) | (resourceIndex << CLUSTER_INDEX_BITS) | clusterIndex;
        return ref;
    }
    public static int completeReferenceMake(int type, int setIndex) {
        assert(type < (1<<TYPE_BITS));
        assert(setIndex < (1<<NOTYPE_BITS));
        int ref = (type << NOTYPE_BITS) | setIndex;
        return ref;
    }
}
