/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.procore.cluster;

import org.simantics.db.procore.cluster.LongHash.AllocatorI;

public class ClusterObjectSet {
	public static final int HeaderSize = LongHash.HeaderSize;
	private final AllocatorI allocator;
	private long[] longs;
	private int hashBase;
	public ClusterObjectSet(AllocatorI allocator) {
		this.allocator = allocator;
		longs = allocator.getLongs();
		hashBase = allocator.getHashBase();
	}
	public ClusterObjectSet(AllocatorI allocator, int initialSize) {
		this.allocator = allocator;
		hashBase = LongHash.setUp(allocator, initialSize);
		longs = allocator.getLongs();
	}
	public boolean add(long a) {
		if (LongHash.add(allocator, a)) {
			longs = allocator.getLongs();
			hashBase = allocator.getHashBase();
			return true;
		}
		return false;
	}
	public void clear() {
		LongHash.clear(longs, hashBase);
	}
	public void compact() {
		LongHash.compact(allocator);
		longs = allocator.getLongs();
		hashBase = allocator.getHashBase();
	}
	public boolean contains(long a) {
		return LongHash.contains(longs, hashBase, a);
	}
	public boolean ensureSize(int size) {
		if (LongHash.ensureSize(allocator, size)) {
			longs = allocator.getLongs();
			hashBase = allocator.getHashBase();
			return true;
		}
		return false;
	}
	public int getCapacity() {
		return LongHash.getRealSize(longs, hashBase);
	}
	public LongIterator getObjeccts() {
		return new LongIterator(allocator);
	}
	public int getSize() {
		return LongHash.getUsedSize(longs, hashBase);
	}
	public boolean isEmpty() {
		return LongHash.isEmpty(longs, hashBase);
	}
	public boolean remove(long a) {
		return LongHash.remove(longs, hashBase, a);
	}
    private static void test(boolean ensureOn, boolean dumpOn, final int addedLongs, final long longOffset) {
        String eo = ensureOn ? "on" : "off";
        System.out.println("********************************************");
        System.out.println("Set test with ensure "+ eo);
        long start = System.nanoTime();
        AllocatorTest allocator = new AllocatorTest(null, 0);
        ClusterObjectSet longSet = new ClusterObjectSet(allocator, 0);
        
        if (ensureOn) {
        	longSet.ensureSize(addedLongs);
        }

        for (long i = 0; i < addedLongs; ++i) {
        	if (!longSet.add(i + longOffset))
        		throw new Error("Add failed.");
        }
        
        double end = (System.nanoTime() - start) / 1000000.0;
        System.out.println("Elapsed time in milliseconds " + end + " for adding " + addedLongs + ".");

        if (dumpOn)
        	allocator.dump();
        
        start = System.nanoTime();
        for (long i = 0; i < addedLongs; ++i) {
        	if (!longSet.contains(i + longOffset))
        		throw new Error("Contain failed.");
        }
        end = (System.nanoTime() - start) / 1000000.0; 
        System.out.println("Elapsed time in milliseconds " + end + " for reading.");

        start = System.nanoTime();
        //int count = 0;
        long next;
        final long longSize = addedLongs + longOffset;
        final long freeLong = LongHash.setFree();
        LongIterator iterator = longSet.getObjeccts();
        while (freeLong != (next = iterator.next())) {
        	//++count;
        	if (next < longOffset || next >= longSize)
        		throw new Error("Iterator failed.");
        }
        end = (System.nanoTime() - start) / 1000000.0; 
        System.out.println("Elapsed time in milliseconds " + end + " for looping.");

        start = System.nanoTime();
        for (long i = 0; i < addedLongs; ++i) {
        	if (!longSet.remove(i + longOffset))
        		throw new Error("Remove failed.");
        } 
        end = (System.nanoTime() - start) / 1000000.0;
        System.out.println("Elapsed time in milliseconds " + end + " for removing.");
        
        if (longSet.getSize() != 0)
    		throw new Error("Element count not zero.");

        if (dumpOn)
        	allocator.dump();

        System.out.println("Real hash space consumption " + longSet.getCapacity());
        start = System.nanoTime();
        longSet.compact();
        end = (System.nanoTime() - start) / 1000000.0;
        System.out.println("Elapsed time in      " + end + " for compact.");
        System.out.println("Real hash space consumption " + longSet.getCapacity());

        if (dumpOn)
        	allocator.dump();

        System.out.println("********************************************");
    }
        
    private static class AllocatorTest implements AllocatorI {
    	private static final int HASH_OFFSET = 10;
    	private long[] longs;
    	private int hashBase;

    	AllocatorTest(long[] longs, int hashBase) {
    	}
    	@Override
		public int allocate(int size) { // return hash base index
			hashBase = HASH_OFFSET + LongHash.HeaderSize;
			longs = new long[size + hashBase];
			return HASH_OFFSET + LongHash.HeaderSize;
		}

		@Override
		public final int getHashBase() {
			return hashBase;
		}

		@Override
		public final long[] getLongs() {
			return longs;
		}
    	
	    void dump() {
	        for(int i=0; i<longs.length; ++i)
	            System.out.println("longs[" + i + "]=" + longs[i]);
	        System.out.println("longs capacity=" + longs.length);
	    }
  }
    public static void main(String[] args) {
    	final boolean dumpOn = true;
    	final int addedLongs = 100; //00000;
    	final long longOffset = 100;
    	test(false, dumpOn, addedLongs, longOffset);
        test(true, dumpOn, addedLongs, longOffset);
    }
   
}
