/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package fi.vtt.simantics.procore.internal;

import java.io.File;

import org.simantics.db.Database;
import org.simantics.db.SessionManager;
import org.simantics.db.SessionReference;
import org.simantics.db.VirtualGraph;
import org.simantics.db.authentication.UserAuthenticationAgent;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.InvalidAuthenticationException;
import org.simantics.db.exception.InvalidUserException;
import org.simantics.db.impl.ClusterI;
import org.simantics.db.impl.ClusterSupport;
import org.simantics.db.impl.ClusterTranslator;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.impl.graph.WriteSupport;
import org.simantics.db.impl.query.QueryProcessor;
import org.simantics.db.impl.query.QuerySupport;
import org.simantics.db.service.ServerInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final public class SessionImplDb extends SessionImplSocket {

    private static final Logger LOGGER = LoggerFactory.getLogger(SessionImplDb.class);

    /**
     * Cached ServerInformation structure fetched from the server at connection
     * time. It should never change during a single session and therefore it
     * should be perfectly safe to cache it.
     */
    protected ServerInformationImpl serverInfo;

    public SessionImplDb(SessionManager sessionManager, UserAuthenticationAgent authAgent) {
        super(sessionManager, authAgent);
    }

    @Override
    protected VirtualGraph getProvider(VirtualGraph vg) {
        return vg;
    }

    @Override
    public ResourceImpl getNewResource() throws DatabaseException {
        if (null != defaultClusterSet)
            return getNewResource(defaultClusterSet);
        ClusterI cluster = getNewResourceCluster();
        int newId;
        try {
            newId = cluster.createResource(clusterTranslator);
        } catch (DatabaseException e) {
            LOGGER.error("createResource failed", e);
            return null;
        }
        return new ResourceImpl(resourceSupport, newId);
    }

    public void connect(SessionReference sessionReference, final Database.Session dbSession) throws Exception {
        if (null == clusterTable) {
            File t = StaticSessionProperties.virtualGraphStoragePath;
            if (null == t)
                t = new File(".");
            clusterTable = new ClusterTable(this, t);
        }

        graphSession = new GraphSessionSocket(this, sessionReference, dbSession);

        try {

            clusterStream = new ClusterStream(this, graphSession, false);

            clusterTranslator = new ClusterTranslatorImpl(this);
            serviceLocator.registerService(ClusterTranslator.class, clusterTranslator);
            serviceLocator.registerService(ClusterSupport.class, clusterTranslator);

            resourceSupport = new ResourceSupportImpl(this);

            requestManager = new SessionRequestManager(this, state);

            querySupport = new QuerySupportImpl(this, clusterTranslator, new SerialisationSupportImpl(this), requestManager);
            serviceLocator.registerService(QuerySupport.class, querySupport);

            queryProvider2 = new QueryProcessor(getAmountOfQueryThreads(), querySupport, sessionThreads);
            
            if("true".equals(System.getProperty("org.simantics.db.persistQueries")))
                queryProvider2.restore();

            writeSupport = new WriteSupportImpl(this);
            serviceLocator.registerService(WriteSupport.class, writeSupport);

            state.setGraphSession(this, graphSession, queryProvider2, clusterTable);

            this.serverInfo = graphSession.getServerInformation();
            
//            authenticator = authAgent.getAuthenticator(serverInfo);
//            if (authenticator == null)
//                throw new InvalidAuthenticationException("Authentication agent did not provide an authenticator");
//            if (authenticator instanceof BackdoorAuthenticator)
//                user = authenticator.getUser(this);

        } catch (InvalidAuthenticationException e) {
            throw e;
        } catch (InvalidUserException e) {
            throw e;
//        } catch (IOException e) {
//            Logger.defaultLogError("I/O error. See exception for details.", e);
//            graphSession = null;
//            clusterTable.dispose(); clusterTable = null;
//            throw e;
        } catch (Throwable e) {
            LOGGER.error("Unhandled error. See exception for details.", e);
            graphSession = null;
            clusterTable.dispose(); clusterTable = null;
            throw new Exception(e);
        }

        String databaseId = serverInfo.getDatabaseId();
        String serverId = serverInfo.getServerId();
        String id = databaseId + "." + serverId;
        virtualGraphServerSupport.connect(id);
        clusterSetsSupport.connect(id);

    }

    @Override
    protected ServerInformation getCachedServerInformation() {
        return serverInfo;
    }

}
