package fi.vtt.simantics.procore.internal;

import java.io.IOException;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.db.Resource;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.RuntimeDatabaseException;
import org.simantics.db.impl.graph.WriteGraphImpl;
import org.simantics.db.procore.cluster.ClusterImpl;
import org.simantics.db.service.ClusterBuilder.ResourceHandle;
import org.simantics.db.service.ClusterBuilder.StatementHandle;
import org.simantics.db.service.SerialisationSupport;

import fi.vtt.simantics.procore.internal.SessionImplSocket.WriteOnlySupport;

final public class ResourceHandleImpl implements ResourceHandle {
	
	final private ClusterImpl cluster;
	final public int resourceKey;
	
	public ResourceHandleImpl(WriteOnlySupport support) throws DatabaseException {
		this.resourceKey = support.createResourceKey(1);
		this.cluster = support.currentCluster;
		if(cluster.cc == null) cluster.cc = new ClusterChange(support.stream, cluster);
	}
	
	public ResourceHandleImpl(ClusterStream stream, ClusterImpl cluster, int resourceKey) {
		this.resourceKey = resourceKey;
		this.cluster = cluster;
    	if(cluster.cc == null) cluster.cc = new ClusterChange(stream, cluster);
	}

	@Override
	public void applyPredicate(Object cluster) {
		
		ClusterImpl impl = (ClusterImpl)cluster;
		impl.change.addStatementIndex1(resourceKey, this.cluster.clusterUID, (byte)0, impl.foreignLookup);
		
	}

	@Override
	public void applyObject(Object cluster) {

		ClusterImpl impl = (ClusterImpl)cluster;
		impl.change.addStatementIndex2(resourceKey, this.cluster.clusterUID, (byte)0, impl.foreignLookup);
		
	}
	
	@Override
	public void addStatement(StatementHandle handle) {

	    if(cluster.getImmutable()) return;

		cluster.change.addStatementIndex0(resourceKey, ClusterChange.ADD_OPERATION);
    	handle.apply(cluster);
		cluster.cc.addChange(cluster.change);
		
	}
	
	@Override
	public void addStatement(ResourceHandle predicate, ResourceHandle object) {
		
	    if(cluster.getImmutable()) return;
	    
		Change change = cluster.change;
		change.addStatementIndex0(resourceKey, ClusterChange.ADD_OPERATION);
    	predicate.applyPredicate(cluster);
    	object.applyObject(cluster);
		cluster.cc.addChange(change);
		
	}
	
	@Override
	public void addStatement(WriteOnlyGraph graph, ResourceHandle predicate, ResourceHandle object) throws DatabaseException {
		if(cluster.isWriteOnly()) addStatement(predicate, object);
		else {
            WriteGraphImpl impl = (WriteGraphImpl)graph;
		    impl.writeSupport.claim(graph.getProvider(), resourceKey, ((ResourceHandleImpl)predicate).resourceKey, ((ResourceHandleImpl)object).resourceKey);
		}
	}
	
	@Override
	public void addValue(WriteOnlyGraph graph, byte[] bytes) throws DatabaseException {

		if(cluster.isWriteOnly()) cluster.cc.setValue((short)(resourceKey & 0xFFFF), bytes);
		else {
		    WriteGraphImpl impl = (WriteGraphImpl)graph;
		    impl.writeSupport.claimValue(null, resourceKey, bytes, bytes.length);
		}
		
	}
	
	@Override
	public void addValue(Object value, Binding binding) {

		try {
			Serializer ser = Bindings.getSerializerUnchecked(binding);
			byte[] bytes = ser.serialize(value);
			cluster.cc.setValue((short)(resourceKey & 0xFFFF), bytes);
		} catch (SerializationException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		
	}
	
	@Override
	public void addValue(Object value, Serializer serializer) {

		try {
			byte[] bytes = serializer.serialize(value);
			cluster.cc.setValue((short)(resourceKey & 0xFFFF), bytes);
		} catch (SerializationException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		
	}
	
	@Override
	public Resource resource(SerialisationSupport ss) {
	    try {
            return ss.getResource(resourceKey);
        } catch (DatabaseException e) {
            throw new RuntimeDatabaseException(e);
        }
	}
	
}
