/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package fi.vtt.simantics.procore.internal;

import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.exception.RuntimeDatabaseException;
import org.simantics.db.impl.ClusterBase;
import org.simantics.db.impl.ClusterI;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.impl.TransientGraph;
import org.simantics.db.procore.cluster.ClusterImpl;

public class ClusteringSupportImpl extends org.simantics.db.impl.ClusteringSupportImpl {

    final private SessionImplSocket session;

    ClusteringSupportImpl(SessionImplSocket session) {
        this.session = session;
    }
    
    @Override
    public int maximumClusterSize() {
        return ClusterTable.CLUSTER_FILL_SIZE;
    }

    @Override
    public long createCluster() {
        long id;
        try {
            id = session.graphSession.newClusterId();
        } catch (DatabaseException e) {
            throw new RuntimeDatabaseException("Failed to get new cluster id.", e);
        }
        session.clusterTable.makeCluster(id, session.writeOnly);
        return id;
    }

    @Override
    public long getCluster(Resource r) {
        int id = session.querySupport.getId(r);
        if(id < 0) // Virtual resource
            return TransientGraph.getVirtualClusterKey(id);
        return session.clusterTable.getClusterIdByResourceKeyNoThrow(id);
    }

    @Override
    public int getNumberOfResources(long clusterId) throws DatabaseException {
        return session.clusterTable.getClusterByClusterId(clusterId).getNumberOfResources(session.clusterTranslator);
    }

    @Override
    public Resource getResourceByKey(int resourceKey)
            throws ResourceNotFoundException {
        return session.getResourceByKey(resourceKey);
    }

    @Override
    public Resource getResourceByIndexAndCluster(int resourceIndex, long clusterId)
            throws DatabaseException, ResourceNotFoundException {
        if (resourceIndex < 1)
            throw new ResourceNotFoundException("Illegal resource index=" + resourceIndex + " cluster=" + clusterId);
        ClusterI cluster = session.getClusterTable().getLoadOrThrow(clusterId);
        int n = cluster.getNumberOfResources(null);
        if (resourceIndex > n)
            throw new ResourceNotFoundException("Illegal resource index=" + resourceIndex + " cluster=" + clusterId
                    + " max index=" + n);
        return session.getResource(resourceIndex, clusterId);
    }

    @Override
    public Resource getClusterSetOfCluster(Resource r) throws DatabaseException {
        if(!r.isPersistent()) return null;
        ClusterImpl cluster = session.clusterTable.getClusterByResourceKey(((ResourceImpl)r).id);
        Long rid = session.clusterSetsSupport.getSet(cluster.getClusterId());
        if(rid == null || rid == 0) return null;
        return session.resourceSerializer.getResource(rid);
    }

    @Override
    public Resource getClusterSetOfCluster(long cluster) throws DatabaseException {
        Long rid = session.clusterSetsSupport.getSet(cluster);
        if(rid == null || rid == 0) return null;
        return session.resourceSerializer.getResource(rid);
    }

    @Override
    public boolean isClusterSet(Resource r) throws DatabaseException {
        return session.containsClusterSet(r);
    }

    @Override
    public long clusterIdByResourceKey(int resourceKey) {
        return session.clusterTable.getClusterIdByResourceKeyNoThrow(resourceKey);
    }

    @Override
    public long clusterIdByClusterKey(int clusterKey) {
        ClusterBase base = session.clusterTable.getClusterByClusterKey(clusterKey);
        return base.clusterId;
    }

    @Override
    public int clusterKeyByClusterId(long clusterId) {
        return session.clusterTable.getClusterKeyByClusterUIDOrMakeProxy(0, clusterId);
    }

    @Override
    public boolean isImmutableClusterKey(int clusterKey) {
        return session.clusterTable.isImmutableClusterKey(clusterKey);
    }

    @Override
    public boolean isImmutableClusterId(long clusterId) {
        return isImmutableClusterKey(clusterKeyByClusterId(clusterId));
    }

}
