/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package fi.vtt.simantics.procore.internal;

import java.io.InputStream;
import java.util.function.Consumer;

import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ClusterI;
import org.simantics.db.impl.ClusterSupport;
import org.simantics.db.impl.ClusterTraitsBase;
import org.simantics.db.impl.ForEachObjectContextProcedure;
import org.simantics.db.impl.ForEachObjectProcedure;
import org.simantics.db.impl.ForPossibleRelatedValueContextProcedure;
import org.simantics.db.impl.ForPossibleRelatedValueProcedure;
import org.simantics.db.impl.Table;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.procore.cluster.ClusterBig;
import org.simantics.db.procore.cluster.ClusterImpl;
import org.simantics.db.procore.cluster.ClusterTraits;
import org.simantics.db.service.ClusterUID;

import fi.vtt.simantics.procore.DebugPolicy;

final public class ClusterWriteOnly extends ClusterImpl {

    final SessionImplSocket sessionImpl;
    final ClusterTable clusterTable;
    private final int clusterKeyHigh;
    private short currentIndex;

    ClusterWriteOnly(ClusterUID clusterUID, int clusterKey, SessionImplSocket sessionImpl) {
        super(clusterUID , clusterKey, sessionImpl.clusterTranslator);
        if(DebugPolicy.REPORT_CLUSTER_EVENTS)
            new Exception(clusterUID.toString()).printStackTrace();
        assert(sessionImpl != null);
        this.sessionImpl = sessionImpl;
        this.clusterKeyHigh = ClusterTraits.getClusterBits(clusterKey);
        this.cc = new ClusterChange(sessionImpl.clusterStream, this);
        clusterTable = sessionImpl.clusterTable;
        currentIndex = 1;
        setImportance(Long.MAX_VALUE);
    }
    boolean isNew() {
        return true;
    }
    @Override
    public void releaseMemory() {
    }
    @Override
    public void compact() {
    }
    @Override
    public boolean isLoaded() {
        return true; // write only cluster is never loaded
    }
    @Override
    public boolean isWriteOnly() {
        return true;
    }
    @Override
    public int createResource(ClusterSupport support) throws DatabaseException {
        cc.createResource(currentIndex);
//        System.err.println("write only resource [" + clusterId + "] " + currentIndex);
        if(DebugPolicy.REPORT_RESOURCE_ID_ALLOCATION)
            System.out.println("[RID_ALLOCATION]: ClusterWriteOnly[" + clusterId + "] allocates " + currentIndex);
        return clusterKeyHigh + currentIndex++;
    }
    @Override
    public boolean hasResource(int r, ClusterSupport support)
    throws DatabaseException {
        int resourceIndex = ClusterTraits.getResourceIndexFromResourceKey(r);
        if (ClusterTraits.getClusterKeyFromResourceKey(r) != this.clusterKey)
            return false;
        if (resourceIndex > 0 && resourceIndex <= currentIndex)
            return true;
        else
            return false;
    }
    private void addChange(int s, int p, int o, ClusterSupport a, byte operation) throws DatabaseException {

        change.op0 = operation;
        change.key0 = s;
        change.key1 = p;
        change.key2 = o;

        if(ClusterTraits.isCluster(clusterKeyHigh, p)) {
            change.clusterUID1 = getClusterUID();
        } else {
            change.clusterUID1 = clusterTable.getClusterUIDByResourceKey(p);
        }

        if(ClusterTraits.isCluster(clusterKeyHigh, o)) {
            change.clusterUID2 = getClusterUID();
        } else {
            change.clusterUID2 = clusterTable.getClusterUIDByResourceKey(o);
        }

        cc.addChange(change);
    }
    @Override
    public ClusterI addRelation(int s, int p, int o, ClusterSupport a) throws DatabaseException {
    	addChange(s, p, o, a, ClusterChange.ADD_OPERATION);
        return this;
    }
    @Override
    synchronized public boolean removeRelation(int s, int p, int o, ClusterSupport a) throws DatabaseException {
    	addChange(s, p, o, a, ClusterChange.REMOVE_OPERATION);
    	return true;
    }
    @Override
    synchronized public void denyRelation(int s, int p, int o, ClusterSupport a) throws DatabaseException {
    	addChange(s, p, o, a, ClusterChange.REMOVE_OPERATION);
    }
    @Override
    synchronized public ClusterI setValue(int s, byte[] value, int length, ClusterSupport a)
    throws DatabaseException {
        sessionImpl.clusterTranslator.addStatementIndex(this, s, getClusterUID(), ClusterStream.SET_OPERATION);
        sessionImpl.clusterTranslator.setValue(this, clusterId, value, length);
        return this;
    }
    @Override
    public ClusterI modiValueEx(int s, long voffset, int length, byte[] value, int offset, ClusterSupport support)
    throws DatabaseException {
        sessionImpl.clusterTranslator.addStatementIndex(this, s, getClusterUID(), ClusterStream.MODI_OPERATION);
        support.modiValue(this, getClusterId(), voffset, length, value, offset);
        return this;
    }
    @Override
    public byte[] readValueEx(int s, long voffset, int length, ClusterSupport support)
    throws DatabaseException {
        throw new Error("Not implemented");
    }
    @Override
    public long getValueSizeEx(int rResourceId, ClusterSupport support)
    throws DatabaseException {
        throw new Error("Not implemented");
    }
    @Override
    synchronized public void setValueEx(int s)
    throws DatabaseException {
    }
    @Override
    synchronized public boolean removeValue(int s, ClusterSupport a) {
        throw new Error("Not implemented");
    }
    @Override
    synchronized public ClusterI getClusterByResourceKey(int resourceKey, ClusterSupport support) {
        int clusterShortId = ClusterTraitsBase.getClusterKeyFromResourceKeyNoThrow(resourceKey);
        if (this.clusterKey == clusterShortId)
            return this;
        return support.getClusterByResourceKey(resourceKey);
    }
    @Override
    synchronized public int getNumberOfResources(ClusterSupport support) {
        return currentIndex - 1;
    }
    @Override
    public boolean isEmpty() {
        return true;
    }
    @Override
    public long getUsedSpace()
    throws DatabaseException {
        return 0;
    }
    @Override
    public void decreaseReferenceCount(int amount) {
    }
    @Override
    public void increaseReferenceCount(int amount) {
    }
    @Override
    public int getReferenceCount() {
        return 1;
    }
    @Override
    public byte[] getValue(int sr, ClusterSupport a) {
        throw new Error("Not implemented.");
    }
    @Override
    public InputStream getValueStream(int resourceKey, ClusterSupport support) throws DatabaseException {
        throw new Error("Not implemented.");
    }
    @Override
    public boolean hasValue(int r, ClusterSupport a) {
        throw new Error("Not implemented.");
    }
    @Override
    public void printDebugInfo(String message, ClusterSupport support) {
        throw new Error("Not implemented.");
    }
    @Override
    public void load() {
    }
    @Override
    public void load(Consumer<DatabaseException> r) {
    }

    @Override
    public int getSingleObject(int resourceKey, int predicateKey, ClusterSupport support) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public <T> int getSingleObject(int resourceKey,
            ForPossibleRelatedValueProcedure<T> procedure,
            ClusterSupport support) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public <C, T> int getSingleObject(int resourceKey,
            ForPossibleRelatedValueContextProcedure<C, T> procedure,
            ClusterSupport support) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public void forObjects(ReadGraphImpl graph, int resourceKey, int predicateKey,
            SyncMultiProcedure<Resource> procedure) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public <Context> boolean forObjects(int resourceKey, int predicateKey, int objectIndex,
            ObjectProcedure<Context> procedure, Context context, ClusterSupport support) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public void forObjects(ReadGraphImpl graph, int resourceKey,
            ForEachObjectProcedure procedure) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public <C> void forObjects(ReadGraphImpl graph, int resourceKey, C context,
            ForEachObjectContextProcedure<C> procedure) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public <Context> boolean forObjects(int resourceKey, int predicateKey,
            ObjectProcedure<Context> procedure, Context context, ClusterSupport support) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public <Context> boolean forPredicates(int resourceKey,
            PredicateProcedure<Context> procedure, Context context, ClusterSupport support) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public int getCompleteObjectKey(int resourceKey, ClusterSupport support) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public CompleteTypeEnum getCompleteType(int resourceKey, ClusterSupport support)
            throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public boolean isComplete(int resourceKey, ClusterSupport support) throws DatabaseException {
        throw new DatabaseException("Not implemented.");
    }
    @Override
    public boolean hasVirtual() {
        return false;
    }
    @Override
    public void load(ClusterSupport session, Runnable callback) {
        throw new Error();
    }
    @Override
    public boolean contains(int resource) {
        throw new Error();
    }

    @Override
    public ClusterTypeEnum getType() {
        return ClusterTypeEnum.WRITEONLY;
    }
    @Override
    public int execute(int valueToModify) throws DatabaseException {
        throw new Error("Not supported");
    }
    @Override
    public ClusterBig toBig(ClusterSupport support) throws DatabaseException {
        throw new Error("Not supported");
    }
    @Override
    public void checkDirectReference(int dr) throws DatabaseException {
        throw new Error("Not supported");
    }
    @Override
    public void checkForeingIndex(int fi) throws DatabaseException {
        throw new Error("Not supported");
    }
    @Override
    public void checkObjectSetReference(int or) throws DatabaseException {
        throw new Error("Not supported");
    }
    @Override
    public boolean getImmutable() {
        return false;
    }
    @Override
    public void setImmutable(boolean immutable, ClusterSupport support) {
        sessionImpl.clusterTranslator.setImmutable(this, immutable);
    }
    @Override
    public boolean getDeleted() {
        return false;
    }
    @Override
    public void setDeleted(boolean deleted, ClusterSupport support) {
        sessionImpl.clusterTranslator.setDeleted(this, deleted);
    }
    @Override
    public void checkValueInit() throws DatabaseException {
        throw new UnsupportedOperationException();
        
    }
    @Override
    public void checkCompleteSetReference(int cr) throws DatabaseException {
        throw new UnsupportedOperationException();
        
    }
    @Override
    public void checkPredicateIndex(int pi) throws DatabaseException {
        throw new UnsupportedOperationException();
        
    }
    @Override
    public void checkValue(int capacity, int index) throws DatabaseException {
        throw new UnsupportedOperationException();
        
    }
    @Override
    public void checkValueFini() throws DatabaseException {
        throw new UnsupportedOperationException();
        
    }
    @Override
    public Table<?> getPredicateTable() {
        throw new UnsupportedOperationException();
    }
    @Override
    public Table<?> getForeignTable() {
        throw new UnsupportedOperationException();
    }
    @Override
    public Table<?> getCompleteTable() {
        throw new UnsupportedOperationException();
    }
    @Override
    public Table<?> getValueTable() {
        throw new UnsupportedOperationException();
    }
    @Override
    public int makeResourceKey(int pRef) throws DatabaseException {
        throw new UnsupportedOperationException();
    }
    @Override
    public Table<?> getObjectTable() {
        throw new UnsupportedOperationException();
    }
}

