package fi.vtt.simantics.procore.internal;

import java.io.InputStream;

import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ClusterBase;
import org.simantics.db.impl.ClusterTranslator;
import org.simantics.db.impl.IClusterTable;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.procore.cluster.ClusterImpl;
import org.simantics.db.service.ClusterUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterTranslatorImpl implements ClusterTranslator {

    private static final Logger LOGGER = LoggerFactory.getLogger(ClusterTranslatorImpl.class);

    final private SessionImplSocket session;
    final private ClusterStream clusterStream;
    final private ClusterTable clusterTable;

    ClusterTranslatorImpl(SessionImplSocket session) {
        this.session = session;
        this.clusterStream = session.clusterStream;
        this.clusterTable = session.clusterTable;
    }

    @Override
    public byte[] getValueEx(int resourceIndex, long clusterId)
    throws DatabaseException {
        return session.graphSession.getResourceValue(resourceIndex, clusterTable.makeClusterUID(clusterId));
    }

    @Override
    public InputStream getValueStreamEx(int resourceIndex, long clusterId)
    throws DatabaseException {
        return session.graphSession.getResourceValueStream(resourceIndex, clusterTable.makeClusterUID(clusterId), 0, 0);
    }

    @Override
    public byte[] getValueEx(int resourceIndex, long clusterId, long voffset, int length)
    throws DatabaseException {
        return session.graphSession.getResourceValue(resourceIndex, clusterTable.makeClusterUID(clusterId), voffset, length);
    }

    public long getValueSizeEx(int resourceIndex, long clusterId)
    throws DatabaseException {
        return session.graphSession.getResourceValueSize(resourceIndex, clusterTable.makeClusterUID(clusterId));
    }
    public int wait4RequestsLess(int limit)
    throws DatabaseException {
        return session.graphSession.wait4RequestsLess(limit);
    }
    @Override
    public int createClusterKeyByClusterUID(ClusterUID clusterUID, long clusterId) {
        return clusterTable.makeProxy(clusterUID, clusterId).getClusterKey();
    }
    @Override
    public int getClusterKeyByClusterUIDOrMake(ClusterUID clusterUID) {
        return clusterTable.getClusterKeyByClusterUIDOrMakeProxy(clusterUID);
    }
    @Override
    public int getClusterKeyByClusterUIDOrMake(long id1, long id2) {
        return clusterTable.getClusterKeyByClusterUIDOrMakeProxy(id1, id2);
    }
    @Override
    public ClusterBase getClusterByClusterUIDOrMake(ClusterUID clusterUID) {
        return clusterTable.getClusterByClusterUIDOrMakeProxy(clusterUID);
    }
    @Override
    final public ClusterBase getClusterByClusterId(long clusterId) {
        try {
            return getClusterByClusterIdOrThrow(clusterId);
        } catch (DatabaseException e) {
            LOGGER.error("Cluster not found by id. cluster id={}", clusterId, e);
            return null;
        }
    }

    @Override
    public ClusterBase getClusterByClusterIdOrThrow(long clusterId) throws DatabaseException {
        ClusterBase cluster =  clusterTable.getClusterByClusterId(clusterId);
        if (null != cluster)
            return cluster;
        return clusterTable.getLoadOrThrow(clusterId);
    }

    @Override
    public ClusterBase getClusterByClusterKey(int clusterKey) {
        ClusterBase proxy = clusterTable.getClusterByClusterKey(clusterKey);
        if (null == proxy)
            return null; // throw new ResourceNotFoundException(id);
        return proxy;
    }

    @Override
    public ClusterBase getClusterByResourceKey(int resourceKey) {

        ClusterBase proxy = clusterTable.getClusterByResourceKey(resourceKey);
        if (null == proxy)
            return null; // throw new ResourceNotFoundException(id);

        return proxy;

    }
    @Override
    public long getClusterIdOrCreate(ClusterUID clusterUID) {
        return clusterTable.getClusterIdOrCreate(clusterUID);
    }
    @Override
    public void createResource(Object cluster, short resourceIndex, long clusterId) {
        ClusterImpl impl = (ClusterImpl)cluster;
        if(impl.cc == null) impl.cc = new ClusterChange(clusterStream, impl);
        if (null == clusterStream) {
            System.out.println("Temporary skip of clusterStream.createResource");
            return;
        } else
            clusterStream.createResource(impl.cc, resourceIndex, clusterTable.makeClusterUID(clusterId));
    }

    @Override
    public void addStatementIndex(Object cluster, int resourceKey, ClusterUID clusterUID, byte op) {
        ClusterImpl impl = (ClusterImpl)cluster;
        if (null == impl.change)
            return;
        clusterStream.addStatementIndex(impl.change, resourceKey, clusterUID, op);
    }

    @Override
    public void addStatement(Object cluster) {
        ClusterImpl impl = (ClusterImpl)cluster;
        if(impl.cc == null) impl.cc = new ClusterChange(clusterStream, impl);
        clusterStream.addStatement(impl.cc, impl.change);
    }


    @Override
    public void cancelStatement(Object cluster) {
        ClusterImpl impl = (ClusterImpl)cluster;
        if(impl.cc == null) impl.cc = new ClusterChange(clusterStream, impl);
        clusterStream.cancelStatement(impl.change);
    }

    @Override
    public void removeStatement(Object cluster) {
        ClusterImpl impl = (ClusterImpl)cluster;
        if(impl.cc == null) impl.cc = new ClusterChange(clusterStream, impl);
        clusterStream.removeStatement(impl.cc, impl.change);
    }

    @Override
    public void cancelValue(Object cluster) {
        ClusterImpl impl = (ClusterImpl)cluster;
        if(impl.cc == null) impl.cc = new ClusterChange(clusterStream, impl);
        clusterStream.cancelValue(impl.change);
    }

    @Override
    public void removeValue(Object cluster) {
        ClusterImpl impl = (ClusterImpl)cluster;
        if(impl.cc == null) impl.cc = new ClusterChange(clusterStream, impl);
        clusterStream.removeValue(impl.cc, impl.change);
    }

    @Override
    public void setValue(Object cluster, long clusterId, byte[] bytes, int length) {
        ClusterImpl impl = (ClusterImpl)cluster;
        if(impl.cc == null) impl.cc = new ClusterChange(clusterStream, impl);
        clusterStream.setValue(impl.cc, impl.change, clusterId, bytes, length);
    }

    @Override
    public void modiValue(Object cluster, long clusterId, long voffset, int length, byte[] bytes, int offset) {
        ClusterImpl impl = (ClusterImpl)cluster;
        if(impl.cc == null) impl.cc = new ClusterChange(clusterStream, impl);
        clusterStream.modiValue(impl.cc, impl.change, clusterId, voffset, length, bytes, offset);
    }

    @Override
    public void setStreamOff(boolean setOff) {
        clusterStream.setOff(setOff);
    }

    @Override
    public boolean getStreamOff() {
        return clusterStream.getOff();
    }

    @Override
    public void setImmutable(Object cluster, boolean immutable) {
        ClusterImpl impl = (ClusterImpl)cluster;
    	clusterTable.markImmutable(impl, immutable);
        if(impl.cc == null) impl.cc = new ClusterChange(clusterStream, impl);
        clusterStream.setImmutable(impl.cc, impl.change, immutable);
    }

    @Override
    public void setDeleted(Object cluster, boolean deleted) {
        ClusterImpl impl = (ClusterImpl)cluster;
        //clusterTable.markDeleted(impl, deleted);
        if (impl.cc == null)
            impl.cc = new ClusterChange(clusterStream, impl);
        clusterStream.setDeleted(impl.cc, impl.change, deleted);
    }

    @Override
    public void undoValueEx(Object cluster, int resourceIndex) {
        ClusterImpl impl = (ClusterImpl)cluster;
        if(impl.cc == null) impl.cc = new ClusterChange(clusterStream, impl);
        clusterStream.undoValueEx(impl.cc, impl.change, resourceIndex);
    }

    @Override
    public ResourceImpl getResource(int id) {
        return new ResourceImpl(session.resourceSupport, id);
    }

    @Override
    public ResourceImpl getResource(int callerThread, int id) {
        assert (id != 0);
        return new ResourceImpl(session.resourceSupport, id);
    }

    @Override
    public Session getSession() {
        return session;
    }

    @Override
    public IClusterTable getClusterTable() {
        return clusterTable;
    }

}
