/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db.layer0.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.ResourceMap;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.SubgraphExtent.ExtentStatus;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2.SeedSpec;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2.SeedSpec.SeedSpecType;
import org.simantics.db.procedure.SyncContextMultiProcedure;
import org.simantics.db.service.DirectQuerySupport;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ConsistsOfProcess {

	private static final Logger LOGGER = LoggerFactory.getLogger(ConsistsOfProcess.class);

	final ArrayList<ConsistsOfProcessEntry> result;
	final Set<ConsistsOfProcessEntry> childrenWithNoName;
	final SyncContextMultiProcedure<ConsistsOfProcessEntry, Resource> structure;
	final SyncContextMultiProcedure<ConsistsOfProcessEntry, Resource> names;

	public static Pair<ArrayList<ConsistsOfProcessEntry>,Set<ConsistsOfProcessEntry>> walk(ReadGraph graph, Collection<SeedSpec> specs, boolean ignoreVirtual) throws DatabaseException {
		return walk(graph, null, specs, ignoreVirtual);
	}

	public static Pair<ArrayList<ConsistsOfProcessEntry>,Set<ConsistsOfProcessEntry>> walk(ReadGraph graph, ResourceMap<ExtentStatus> status, Collection<SeedSpec> specs, boolean ignoreVirtual) throws DatabaseException {

		ConsistsOfProcess process = new ConsistsOfProcess(graph, status, specs, ignoreVirtual);
		return Pair.make(process.result, process.childrenWithNoName);

	}

	static class ConsistsOfProcessEntry {
		public ConsistsOfProcessEntry parent;
		public Resource resource;
		public boolean valid = true;
		public String name = null;
		ConsistsOfProcessEntry(ConsistsOfProcessEntry parent, Resource resource) {
			this.parent = parent;
			this.resource = resource;
		}
	}

	private ConsistsOfProcess(ReadGraph graph, ResourceMap<ExtentStatus> status, final Collection<SeedSpec> seeds, final boolean ignoreVirtual) throws DatabaseException {

		final Layer0 L0 = Layer0.getInstance(graph);
		final DirectQuerySupport dqs = graph.getService(DirectQuerySupport.class);

		result = new ArrayList<>();
		childrenWithNoName = new HashSet<>();
		names = dqs.compileForEachObject(graph, L0.HasName, new SyncContextMultiProcedure<ConsistsOfProcessEntry, Resource>() {

			@Override
			public void execute(ReadGraph graph, ConsistsOfProcessEntry entry, Resource nameResource) throws DatabaseException {

				if(status != null)
					status.put(nameResource, ExtentStatus.EXCLUDED);

				if(!entry.valid)
					return;

				String name = graph.getValue(nameResource, Bindings.STRING);
				if(name == null) {
					entry.valid = false;
				} else if (entry.name != null) {
					entry.valid = false;
				} else {
					Resource t = graph.getPossibleObject(nameResource, L0.InstanceOf);
					if(L0.String.equals(t)) {
						entry.name = name;
					} else {
						entry.name = name;
						entry.valid = false;
						status.remove(nameResource);
					}
				}

			}

			@Override
			public void exception(ReadGraph graph, Throwable throwable) {
				LOGGER.error("Exception while resolving ConsistsOf hierarchy", throwable);
			}

			@Override
			public void finished(ReadGraph graph, ConsistsOfProcessEntry entry) {
				if(entry.valid) {
					if(entry.name != null) {
						result.add(entry);
					} else {
						// This one did not have a name - not a valid internal
						childrenWithNoName.add(entry);
					}
				} else {
					// Something wrong has happened. Do not treat as valid internal
					childrenWithNoName.add(entry);
				}
			}
		});

		structure = dqs.compileForEachObject(graph, L0.ConsistsOf, new SyncContextMultiProcedure<ConsistsOfProcessEntry, Resource>() {

			@Override
			public void execute(ReadGraph graph, ConsistsOfProcessEntry parent, Resource child) {

				if(status != null)
					if(ExtentStatus.EXCLUDED.equals(status.get(child))) return;

				if(!ignoreVirtual || child.isPersistent()) {
					ConsistsOfProcessEntry entry = new ConsistsOfProcessEntry(parent, child);
					dqs.forEachObjectCompiled(graph, child, entry, structure);
					dqs.forEachObjectCompiled(graph, child, entry, names);
				}

			}

			@Override
			public void finished(ReadGraph graph, ConsistsOfProcessEntry parent) {
			}

			@Override
			public void exception(ReadGraph graph, Throwable throwable) {
				LOGGER.error("Exception while resolving ConsistsOf hierarchy", throwable);
			}

		});

		graph.syncRequest(new ReadRequest() {

			@Override
			public void run(ReadGraph graph) throws DatabaseException {
				for(SeedSpec seed  : seeds) {

					if(status != null) {
						ExtentStatus es = status.get(seed.resource);
						if(ExtentStatus.EXCLUDED.equals(es)) continue;
						if(ExtentStatus.EXTERNAL.equals(es)) continue;
					}

					ConsistsOfProcessEntry entry = new ConsistsOfProcessEntry(null, seed.resource);

					dqs.forEachObjectCompiled(graph, seed.resource, entry, structure);
					if(SeedSpecType.INTERNAL.equals(seed.specType)) {
						// Process names only for internal seeds
						dqs.forEachObjectCompiled(graph, seed.resource, entry, names);
					}
				}
			}

		});

	}

}
