/*******************************************************************************
 * Copyright (c) 2007, 2025 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #1214
 *******************************************************************************/
package org.simantics.db.layer0.adapter.impl;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.ModelTransferableGraphSource;
import org.simantics.db.layer0.util.ModelTransferableGraphSourceRequest;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2;
import org.simantics.db.service.DirectQuerySupport;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.graph.db.TransferableGraphSource.TransferableGraphSourceProcedure;
import org.simantics.graph.representation.Identity;
import org.simantics.graph.representation.Internal;
import org.simantics.graph.representation.Root;
import org.simantics.layer0.Layer0;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Tuukka Lehtonen
 */
public class TGRemover extends AbstractRemover {

    private static final Logger LOGGER = LoggerFactory.getLogger(TGRemover.class);

    @SuppressWarnings("unused")
    private IProgressMonitor    monitor;

    private ArrayList<Resource> roots = new ArrayList<>();

    public TGRemover(Resource resource) {
        super(resource);
    }

    public TGRemover(IProgressMonitor monitor, Resource resource) {
        super(resource);
        this.monitor = monitor;
    }

    public List<Resource> getRoots() {
        return roots;
    }

    @Override
    public void remove(final WriteGraph graph) throws DatabaseException {

        Layer0 L0 = Layer0.getInstance(graph);

        TransferableGraphConfiguration2 conf = new TransferableGraphConfiguration2(graph, resource);
        conf.values = false;
        conf.valueIds = true;
        final SerialisationSupport ss = graph.getService(SerialisationSupport.class);
        final DirectQuerySupport dqs = graph.getService(DirectQuerySupport.class);

        try (ModelTransferableGraphSource source = graph.syncRequest(new ModelTransferableGraphSourceRequest(conf))) {

            int[] rev = source.getTransientResourceArray(graph);

            source.init(graph);
            var identityCount = source.getIdentityCount();
            LOGGER.trace("removing {} identities", identityCount);
            source.forIdentities(graph, new TransferableGraphSourceProcedure<Identity>() {
                @Override
                public void execute(Identity value) throws Exception {
                    if (value.definition instanceof Internal) {
                        int res = rev[value.resource];
                        Resource r = ss.getResource(res);
                        Resource name = graph.getPossibleObject(r, L0.HasName);
                        if (name != null) {
                            graph.deny(r, L0.HasName, L0.NameOf, name);
                            graph.denyValue(name);
                            for (Statement stm : dqs.getDirectPersistentStatements(graph, name)) {
                                graph.deny(name, stm.getPredicate(), stm.getObject());
                            }
                        }
                    } else if (value.definition instanceof Root) {
                        int res = rev[value.resource];
                        Resource r = ss.getResource(res);
                        roots.add(r);
                    }
                    // Ignore External and Optional identities.
                }
            });

            var stmCount = source.getStatementCount();
            LOGGER.trace("removing {} statements", stmCount);
            source.forResourceStatements(graph, new TransferableGraphSourceProcedure<int[]>() {

                @Override
                public void execute(int[] value) throws Exception {
                    Resource s = ss.getResource(value[0]);
                    Resource p = ss.getResource(value[1]);
                    Resource i = null;
                    if (value[2] != -1)
                        i = ss.getResource(value[2]);
                    Resource o = ss.getResource(value[3]);
                    graph.deny(s, p, i, o);
                }

            });

            var valueCount = source.getValueCount();
            LOGGER.trace("removing {} values", valueCount);
            source.forValueResources(graph, new TransferableGraphSourceProcedure<int[]>() {
                @Override
                public void execute(int[] value) throws Exception {
                    graph.denyValue(ss.getResource(value[0]));
                }
            });
            LOGGER.trace("remove completed");

        } catch (Exception e) {
            throw new DatabaseException(e);
        }

    }

}
