/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.util;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.container.DataContainer;
import org.simantics.databoard.container.DataContainers;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.PasteHandler;
import org.simantics.db.layer0.internal.SimanticsInternal;
import org.simantics.db.layer0.util.SimanticsClipboard.Representation;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.request.Read;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.utils.datastructures.hints.IHintContext.Key;

public class ClipboardUtils {
	
	public static String HINT_TARGET_RESOURCE = "HINT_TARGET_RESOURCE";

	public static <T> T accept(Set<Representation> set, Key key) throws DatabaseException {
		return accept(set, key, Collections.emptyMap());
	}

	public static <T> T accept(Set<Representation> set, Key key, Map<String,Object> hints) throws DatabaseException {
		for(Representation r : set) if(r.getKey().equals(key)) return r.getValue(SimanticsInternal.getSession(), hints);
		return null;
	}
	
	public static <T> T accept(RequestProcessor processor, Set<Representation> set, Key key) throws DatabaseException {
		return accept(processor, set, key, Collections.emptyMap());
	}

	public static <T> T accept(RequestProcessor processor, Set<Representation> set, Key key, Map<String,Object> hints) throws DatabaseException {
		for(Representation r : set) if(r.getKey().equals(key)) return r.getValue(processor, hints);
		return null;
	}

	public static <T> T accept(ReadGraph graph, Set<Representation> set, Key key) throws DatabaseException {
		return accept(graph, set, key, Collections.<String, Object>emptyMap());
	}

	public static <T> T accept(ReadGraph graph, Set<Representation> set, Key key, Map<String,Object> hints) throws DatabaseException {
		for(Representation r : set) if(r.getKey().equals(key)) return r.getValue(graph, hints);
		return null;
	}
	
	public static <T> T acceptPossible(ReadGraph graph, Key key) throws DatabaseException {
		return acceptPossible(graph, key, Collections.emptyMap());
	}

	public static <T> T acceptPossible(ReadGraph graph, Key key, Map<String,Object> hints) throws DatabaseException {
	    Collection<T> results = accept(graph, key, hints);
	    if(results.size() == 1) return results.iterator().next();
	    else return null;
	}

    @SuppressWarnings("unchecked")
    public static <T> Collection<T> accept(ReadGraph graph, Key key, Map<String,Object> hints) throws DatabaseException {
        ArrayList<T> result = new ArrayList<T>();
        for(Set<Representation> rs : SimanticsInternal.getClipboard().getContents()) {
            for(Representation r : rs) {
                if(r.getKey().equals(key)) {
                    result.add((T)r.getValue(graph, hints));
                    continue;
                }
            }
        }
        return result;
    }

	public static SimanticsClipboard fileClipboard(String fileName) throws IOException {
		
		DataContainer container = DataContainers.readFile(new File(fileName));
		
		try {
			Representation tgRep = new TGRepresentation((TransferableGraph1)container.content.getValue(TransferableGraph1.BINDING));
			Representation dcRep = new DataContainerRepresentation(container);
			return SimanticsClipboardImpl.make(dcRep, tgRep);
		} catch (AdaptException e) {
			Logger.defaultLogError(e);
			return SimanticsClipboardImpl.EMPTY;
		}
		
		
	}
	
	public static Representation createTransferableGraph(Resource ... resources) {
		return new TGRepresentation(resources);
	}

	public static Representation createTransferableGraph(TransferableGraphConfiguration2 configuration) {
		return new TGRepresentation(configuration);
	}

	public static Representation createTransferableGraph(boolean ignoreVirtualResources, Resource ... resources) {
		return new TGRepresentation(ignoreVirtualResources, resources);
	}

	public static Representation createVariable(String uri) {
		return new VariableRepresentation(uri);
	}

	public static Representation createVariable(RequestProcessor processor, final Variable var) {
		
		try {
			
			return new VariableRepresentation(processor.syncRequest(new Read<String>() {

				@Override
				public String perform(ReadGraph graph) throws DatabaseException {
					return var.getURI(graph);
				}
				
			}));
			
		} catch (DatabaseException e) {
			Logger.defaultLogError(e);
		}
		
		return null;
		
	}

    public static Representation createCopyResources(final Collection<Resource> resources) {
        return new ResourceCopyRepresentation(resources);
    }

    public static Representation createCutResources(final Collection<Resource> resources) {
        return new ResourceCutRepresentation(resources);
    }
    
    public static PasteHandler pasteHandler(Resource resource) throws DatabaseException {
    	return SimanticsInternal.sync(new ResourceRead<PasteHandler>(resource) {

			@Override
			public PasteHandler perform(ReadGraph graph) throws DatabaseException {
				return graph.adapt(resource, PasteHandler.class);
			}
    		
    	});
    }

}
