/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.migration;

import java.util.ArrayList;

import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.graph.query.Path;
import org.simantics.graph.query.TransferableGraphConversion;
import org.simantics.graph.query.UriUtils;
import org.simantics.graph.refactoring.GraphRefactoringException;
import org.simantics.graph.refactoring.GraphRefactoringUtils;
import org.simantics.graph.refactoring.MappingSpecification;
import org.simantics.graph.refactoring.MappingSpecification.MappingRule;
import org.simantics.graph.representation.Identity;
import org.simantics.graph.representation.Internal;
import org.simantics.graph.representation.TransferableGraph1;
import org.simantics.graph.store.IdentityStore;
import org.simantics.layer0.Layer0;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.set.hash.TIntHashSet;

public class NamespaceMigrationStep implements MigrationStep {
    private static final Logger LOGGER = LoggerFactory.getLogger(NamespaceMigrationStep.class);

    final ArrayList<MappingRule> rules;
    
	public NamespaceMigrationStep(ReadGraph graph, Resource step) throws DatabaseException {
		rules = new ArrayList<MappingRule>();
		Layer0 L0 = Layer0.getInstance(graph);
		for(Resource prefix : ListUtils.toList(graph, step)) {
			String base = graph.getPossibleRelatedValue(prefix, L0.NamespaceMigrationStep_Prefix_base, Bindings.STRING);
			String from = graph.getPossibleRelatedValue(prefix, L0.NamespaceMigrationStep_Prefix_from, Bindings.STRING);
			String to = graph.getPossibleRelatedValue(prefix, L0.NamespaceMigrationStep_Prefix_to, Bindings.STRING);
			if(from != null && to != null) {
				Path fromURI = UriUtils.uriToPath(base+from);
				Path toURI = UriUtils.uriToPath(base+to);
				if(fromURI != null && toURI != null)
					rules.add(new MappingRule(fromURI, toURI));
				else
				    LOGGER.error("Namespace migration uri formation error: base " + base + " from " + from + " to " + to);
			}
		}
	}

    @Override
    public void applyTo(IProgressMonitor monitor, Session session, MigrationState state) throws DatabaseException {
    	
        TransferableGraph1 tg = MigrationUtils.getTG(session, state); 
                
        try {
            MappingSpecification mappingSpec = new MappingSpecification(rules);
            boolean fixed = GraphRefactoringUtils.fixIncorrectRoot(tg.identities);
            LOGGER.info("fixed=" + fixed);
            IdentityStore idStore = TransferableGraphConversion.extractIdentities(tg);
            // Mark internal identities new
            for(Identity id : tg.identities)
                if(id.definition instanceof Internal)
                    idStore.markNew(id.resource);
//            idStore.printChildMap();
//            System.err.println("ids: " + idStore);
//            System.err.println("rc: " + tg.resourceCount);
//            System.err.println("idStore: " + idStore.toArray().length);
            TIntHashSet parentsAffected = new TIntHashSet(); 
            GraphRefactoringUtils.refactor(tg, idStore, mappingSpec, parentsAffected);
//
            tg.resourceCount = idStore.getResourceCount();
            
            tg.identities = idStore.toArray();

            if(fixed)
                GraphRefactoringUtils.unfixIncorrectRoot(tg.identities);

            LOGGER.info("rc2: " + tg.resourceCount);
//            System.err.println("idStore2: " + idStore.toArray().length);

        } catch (GraphRefactoringException e) {
            e.printStackTrace();
        }
    	
    }
    
}
