/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.genericrelation.combinators;

import gnu.trove.list.array.TIntArrayList;

import java.util.ArrayList;
import java.util.List;

import org.simantics.db.ReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.GenericRelation;
import org.simantics.db.layer0.genericrelation.AbstractRelation;
import org.simantics.utils.datastructures.Pair;

/**
 * Selects tuples from the relation by fixing certain fields.
 * Also projects the relation to unfixed fields. 
 * 
 * Do not use this class directly, because relations have often
 * optimized selection methods, but use IRelation.select method.
 */
public class Selection extends AbstractRelation {
	
	AbstractRelation base;
	int baseLength;
	int[] constantPositions;
	int[] variablePositions;
	Object[] constants;
	
	public Selection(AbstractRelation base, String bindingPattern,
			Object[] constants) {
		this.base = base;
		TIntArrayList constantPositionArray = new TIntArrayList();
		TIntArrayList variablePositionsArray = new TIntArrayList();
		baseLength = bindingPattern.length();
		for(int i=0;i<baseLength;++i)
			switch(bindingPattern.charAt(i)) {
			case 'b': constantPositionArray.add(i); break;
			case 'v': variablePositionsArray.add(i); break;
			}
		constantPositions = constantPositionArray.toArray();
		variablePositions = variablePositionsArray.toArray();
		this.constants = constants;
	}

	@Override
	public boolean contains(ReadGraph g, Object[] tuple) throws DatabaseException {
		assert(tuple.length == variablePositions.length);
		Object[] temp = new Object[baseLength];
		for(int i=0;i<constantPositions.length;++i)
			temp[constantPositions[i]] = constants[i];
		for(int i=0;i<variablePositions.length;++i)
			temp[variablePositions[i]] = tuple[i];
		return base.contains(g, temp);
	}

	@Override
	public boolean isRealizable() {
		return base.isRealizable();
	}

	@Override
	public List<Object[]> realize(ReadGraph g) throws DatabaseException {
		ArrayList<Object[]> result = new ArrayList<Object[]>(); 
		loop:
		for(Object[] tuple : base.realize(g)) {
			for(int i=0;i<constantPositions.length;++i)
				if(!constants[i].equals(tuple[constantPositions[i]]))
					continue loop;
			Object[] temp = new Object[variablePositions.length];
			for(int i=0;i<variablePositions.length;++i)
				temp[i] = tuple[variablePositions[i]];
			result.add(temp);
		}
		return result;
	}

	@Override
	public GenericRelation select(String bindingPattern, Object[] givenConstants) {
		assert(bindingPattern.length() == variablePositions.length);
		char[] baseBindingPattern = new char[baseLength];
		Object[] baseConstants = new Object[baseLength];
		for(int i=0;i<constantPositions.length;++i) {
			baseBindingPattern[constantPositions[i]] = 'b';
			baseConstants[constantPositions[i]] = constants[i];
		}
		int j = 0;
		for(int i=0;i<variablePositions.length;++i) {
			char c = baseBindingPattern[variablePositions[i]] = bindingPattern.charAt(i);
			if(c == 'b')
				baseConstants[constantPositions[i]] = givenConstants[j++];
		}
		assert(j == givenConstants.length);
		return base.select(new String(baseBindingPattern), constants);
	}

	@SuppressWarnings("unchecked")
    @Override
	public Pair<String, String>[] getFields() {
		Pair<String, String>[] baseFields = base.getFields();
		Pair<String, String>[] result = new Pair[variablePositions.length];
		for(int i=0;i<variablePositions.length;++i)
			result[i] = baseFields[variablePositions[i]];
		return result;
	}	
	
}
