/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.changeset;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;

import org.simantics.db.ChangeSetIdentifier;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.CommitMetadata;
import org.simantics.db.common.UndoMetadata;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.genericrelation.DependencyChanges;
import org.simantics.db.layer0.genericrelation.DependencyChanges.Change;
import org.simantics.db.layer0.request.IsLinkedTo;
import org.simantics.db.service.ManagementSupport;

public class MetadataUtils {
	
	public static final boolean DEBUG = false;
	public static final boolean DEBUG_IDS = false;

	public static Map<Long, Map<String,byte[]>> getMetadatas(ReadGraph graph, ManagementSupport support, long fromRevisionId, long toRevisionId) throws DatabaseException {

	    TreeMap<Long, Map<String,byte[]>> result = new TreeMap<Long, Map<String,byte[]>>(); 
	            
	    Collection<ChangeSetIdentifier> csis = support.getChangeSetIdentifiers(fromRevisionId, toRevisionId);
        for(ChangeSetIdentifier csi : csis) {
            if(csi.getId() < fromRevisionId) {
                System.err.println("ManagementSupport.getChangeSetIdentifiers returned identifiers outside of the requested range.");
                continue;
            }
            Map<String,byte[]> metadata = csi.getMetadata();
            if(metadata != null) result.put(csi.getId(), metadata);
        }
        long toBeRevision = support.getHeadRevisionId()+1;
        if(toBeRevision >= fromRevisionId && toBeRevision <= toRevisionId) {
            if(graph instanceof WriteGraph) {
                WriteGraph writeGraph = (WriteGraph)graph;
                Map<String,byte[]> metadata = writeGraph.getMetadata();
                if(metadata != null)
                    result.put(toBeRevision, metadata);
            }
        }
        
        return result;
	    
	}
	
	public static Map<Resource, Collection<Change>> getDependencyChangesFrom(ReadGraph graph, long revisionId) throws DatabaseException {
		ManagementSupport support = graph.getService(ManagementSupport.class);
		
        Map<Resource, Collection<Change>> result = new HashMap<Resource, Collection<Change>>();
		Map<Long, Map<String,byte[]>> metadatas = getMetadatas(graph, support, revisionId, Long.MAX_VALUE);
		
		for(Map.Entry<Long, Map<String,byte[]>> e : metadatas.entrySet()) {
		    long csid = e.getKey();
			Map<String,byte[]> metadata = e.getValue();
			if(metadata != null) {
				byte[] changesData = metadata.get(DependencyChanges.class.getName());
				if(changesData != null && changesData.length > 0) {
					DependencyChanges changes = DependencyChanges.deserialise(graph.getSession(), changesData);
					if (changes == null)
						continue;
					if(DEBUG_IDS) System.err.println("-DependencyChanges[" + csid + "]");
					for(Map.Entry<Resource, Change[]> entry : changes.modelChanges.entrySet()) {
						Collection<Change> list = result.get(entry.getKey());
						if(list == null) {
							list = new ArrayList<Change>();
							result.put(entry.getKey(), list);
						}
						for(Change c : entry.getValue()) {
							if(DEBUG) System.err.println("-" + c);
							list.add(c);
						}
					}
				}
			}
		}
		return result;
	}
    
	public static void visitDependencyChangesFrom2(ReadGraph graph,
			Resource model, Change[] changes, ChangeVisitor changeVisitor) throws DatabaseException {
    	for(Change change : changes) {
    		changeVisitor.visit(graph, change, false);
    	}
	}

    /**
     * Finds all changes made to the given {@code model} from the given revision and calls 
     * the {@code visitor} for them.
     */
    public static void visitDependencyChangesFrom(ReadGraph graph, Resource model, long fromRevision,
            ChangeVisitor visitor) throws DatabaseException {
        visitDependencyChangesBetween(graph, model, fromRevision, Long.MAX_VALUE, visitor, false);
    }

    private static void visitDependencyChangesBetween(ReadGraph graph, Resource model, long fromRevision, long toRevision,
            ChangeVisitor visitor, boolean inverted) throws DatabaseException {
        ManagementSupport support = graph.getService(ManagementSupport.class);
//        System.err.println("visitDependencyChangesBetween " + fromRevision + " " + toRevision);
//        Collection<ChangeSetIdentifier> csis = support.getChangeSetIdentifiers(fromRevision, toRevision);
        
        Map<Long, Map<String,byte[]>> metadatas = getMetadatas(graph, support, fromRevision, toRevision);

//        Map<Resource, Collection<Change>> result = new HashMap<Resource, Collection<Change>>();
//        Map<Long, Map<String,byte[]>> metadatas = getMetadatas(graph, support, revisionId, Long.MAX_VALUE);
        
        for(Map.Entry<Long, Map<String,byte[]>> e : metadatas.entrySet()) {

            Map<String,byte[]> metadata = e.getValue();

            browseChanges: {
                byte[] changesData = metadata.get(DependencyChanges.class.getName());
                if(changesData == null || changesData.length == 0) 
                    break browseChanges;
                DependencyChanges changes = DependencyChanges.deserialise(graph.getSession(), changesData);
                if (changes == null)
                    break browseChanges;
                for(Map.Entry<Resource, Change[]> entry : changes.modelChanges.entrySet())
                    if(graph.syncRequest(new IsLinkedTo(model, entry.getKey()),
                            TransientCacheListener.<Boolean>instance()))
                        for(Change change : entry.getValue())
                            visitor.visit(graph, change, inverted);
            }
            
            byte[] undoMetadata = metadata.get(UndoMetadata.class.getName());
            if(undoMetadata != null && undoMetadata.length != 0) { 
                UndoMetadata undo = UndoMetadata.deserialise(graph.getSession(), undoMetadata);
                visitDependencyChangesBetween(graph, model, undo.getBeginCSId(), undo.getEndCSId(), visitor, 
                        undo.isRedo() ? inverted : !inverted);
            }
        }
    }
    
    public static String printChanges(ReadGraph graph, long fromRevision, long toRevision) throws DatabaseException {
        ManagementSupport support = graph.getService(ManagementSupport.class);
        Map<Long, Map<String,byte[]>> metadatas = getMetadatas(graph, support, fromRevision, toRevision);
        
        StringBuilder sb = new StringBuilder();
        
        for(Map.Entry<Long, Map<String,byte[]>> e : metadatas.entrySet()) {

            Map<String,byte[]> metadata = e.getValue();
            sb.append("R: " + e.getKey());
            sb.append("\n");
            for (Entry<String, byte[]> en : metadata.entrySet()) {
                sb.append("   " + en.getKey() + " " + (en.getValue() == null ? "null" : en.getValue().length));
                sb.append("\n");
            }
            byte[] commitData = metadata.get(CommitMetadata.class.getName());
            if (commitData != null && commitData.length > 0) {
                CommitMetadata commitMetadata = CommitMetadata.deserialise(graph.getSession(), commitData);
                sb.append("   Commit: " + commitMetadata.opid + " " + commitMetadata.date);
            }
            byte[] commentData = metadata.get(CommentMetadata.class.getName());
            if (commentData != null && commentData.length > 0) {
                CommentMetadata commentMetadata = CommentMetadata.deserialise(graph.getSession(), commentData);
                sb.append("   Comment: " + commentMetadata.toString());
            }
            byte[] changesData = metadata.get(DependencyChanges.class.getName());
            if(changesData != null && changesData.length > 0) {
                DependencyChanges changes = DependencyChanges.deserialise(graph.getSession(), changesData);
                for(Map.Entry<Resource, Change[]> entry : changes.modelChanges.entrySet()) {
                     sb.append("   " + entry.getKey());
                     sb.append("\n");
                     for (Change c : entry.getValue()) {
                         sb.append("      " + c.getClass().getSimpleName() + " " + c.getType());
                         sb.append("\n");
                     }
                }
            }
            
        }
        return sb.toString();
        
    }

}
