/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy  - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.impl.ClusterTraitsBase;
import org.simantics.db.impl.ClusteringSupportImpl;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.request.QueryFactoryKey;
import org.simantics.db.request.QuerySerializer;
import org.simantics.db.service.ClusteringSupport;

import gnu.trove.list.array.TByteArrayList;
import gnu.trove.map.hash.TLongLongHashMap;
import gnu.trove.procedure.TLongLongProcedure;

public class QuerySerializerImpl implements QuerySerializer {

	public QueryProcessor processor;
	private QuerySupport querySupport;
	private ClusteringSupportImpl clusteringSupport;
	private TByteArrayList bytes = new TByteArrayList();
	private TLongLongHashMap clusterInfo = new TLongLongHashMap();
	private Map<String,Integer> ids = new HashMap<String,Integer>();
	private long cacheId;
	private boolean immutable;

	public QuerySerializerImpl(QueryProcessor processor, long cacheId, boolean immutable) {
		this.processor = processor;
		this.querySupport = processor.querySupport;
		this.clusteringSupport = (ClusteringSupportImpl)processor.getSession().getService(ClusteringSupport.class);
		this.cacheId = cacheId;
		this.immutable = immutable;
	}
	
	public boolean isImmutable() {
		return immutable;
	}

	public int writeUnknownSize() {
		int pos = bytes.size();
		bytes.add((byte)0);
		bytes.add((byte)0);
		bytes.add((byte)0);
		bytes.add((byte)0);
		return pos;
	}

	public void setUnknownSize(int pos, int value) {
		bytes.set(pos, (byte) (value & 0xFF));
		bytes.set(pos+1, (byte) ((value >>> 8) & 0xFF));
		bytes.set(pos+2, (byte) ((value >>> 16) & 0xFF));
		bytes.set(pos+3, (byte) ((value >>> 24) & 0xFF));
	}

	public void serializeId(QueryFactoryKey qfk) {
		String key = qfk.key();
		Integer id = ids.get(key);
		if(id == null) {
			id = ids.size() + 1;
			ids.put(key, id);
		}
		addLE4(id);
	}

	public void addResource(int r) {
		if(r < 0) {
			addLE4(r);
		} else if (r == 0) {
			addLE2((short)0);
		} else {
			long clusterId = querySupport.getClusterId(r);
			long info = clusterInfo.get(clusterId);
			if(info == 0) {
				boolean immu = clusteringSupport.isImmutableClusterId(clusterId);
				info = ((clusterInfo.size() + 1) << 1) + (immu ? 1:0);
				clusterInfo.put(clusterId, info);
			}
			int clusterKey = (int)(info >> 1);
			int i = ClusterTraitsBase.createResourceKeyNoThrow(clusterKey, ClusterTraitsBase.getResourceIndexFromResourceKeyNoThrow(r));
			addLE4(i);
		}
	}

	public void addResource(Resource resource) {
		if(resource == null) {
			addResource(0);
		} else {
			addResource(((ResourceImpl)resource).id);
		}
	}
	
	public void addStatement(Statement stm) {
		addResource(stm.getSubject());
		addResource(stm.getPredicate());
		addResource(stm.getObject());
	}

	public long cluster(Resource resource) {
		return processor.cluster(((ResourceImpl)resource).id);
	}

	public void addException() {
		bytes.add((byte)0);
		bytes.add((byte)0);
	}

	public void addString(String s) {
		byte[] b = s.getBytes();
		addLE4(b.length);
		bytes.add(b);
	}

	public void add(byte b) {
		bytes.add(b);
	}

	public void add(byte[] bs) {
		bytes.add(bs);
	}

	public byte[] bytes() {
		
		TByteArrayList header = new TByteArrayList();

		writeLE8(header, cacheId);
		header.add(immutable ? (byte)1 : (byte)0);

		writeLE4(header, ids.size());
		for(Entry<String,Integer> entry : ids.entrySet()) {
			String id = entry.getKey();
			writeLE4(header, id.length());
			header.add(id.getBytes());
			writeLE4(header, entry.getValue());
		}

		writeLE4(header, clusterInfo.size());
		clusterInfo.forEachEntry(new TLongLongProcedure() {

			@Override
			public boolean execute(long clusterId, long info) {
				writeLE8(header, clusterId);
				int key = (int)(info >> 1);
				writeLE4(header, key);
				byte immu = (byte)(info & 1);
				header.add(immu);
				return true;
			}

		});

		header.add(bytes.toArray());
		return header.toArray();
	}

	public void addLE4(int value) {
		writeLE4(bytes, value);
	}

	public void addLE2(short value) {
		writeLE2(bytes, value);
	}

	public static void writeLE4(TByteArrayList bytes, int value) {
		bytes.add((byte) (value & 0xFF));
		bytes.add((byte) ((value >>> 8) & 0xFF));
		bytes.add((byte) ((value >>> 16) & 0xFF));
		bytes.add((byte) ((value >>> 24) & 0xFF));
	}

	public static void writeLE2(TByteArrayList bytes, short value) {
		bytes.add((byte) (value & 0xFF));
		bytes.add((byte) ((value >>> 8) & 0xFF));
	}

	public void writeLE8(long value) {
		writeLE8(bytes, value);
	}

	public static void writeLE8(TByteArrayList bytes, long value) {
		bytes.add((byte) (value & 0xFF));
		bytes.add((byte) ((value >>> 8) & 0xFF));
		bytes.add((byte) ((value >>> 16) & 0xFF));
		bytes.add((byte) ((value >>> 24) & 0xFF));
		bytes.add((byte) ((value >>> 32) & 0xFF));
		bytes.add((byte) ((value >>> 40) & 0xFF));
		bytes.add((byte) ((value >>> 48) & 0xFF));
		bytes.add((byte) ((value >>> 56) & 0xFF));
	}

	public QueryProcessor getQueryProcessor() {
		return processor;
	}

	@Override
	public void addByte(byte b) {
		bytes.add(b);
	}
	
	@Override
	public void addBytes(byte[] bs) {
		addLE4(bs.length);
		bytes.add(bs);
	}

}
