/*******************************************************************************
 * Copyright (c) 2018, 2025 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.eclipse.core.runtime.Platform;
import org.simantics.db.impl.ClusterTraitsBase;
import org.simantics.db.impl.ClusteringSupportImpl;
import org.simantics.db.impl.graph.WriteGraphImpl;
import org.simantics.db.request.AsyncMultiRead;
import org.simantics.db.request.AsyncRead;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.PersistentRead;
import org.simantics.db.request.Read;
import org.simantics.db.service.ClusteringSupport;
import org.simantics.utils.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueryCaches {

	private static final Logger LOGGER = LoggerFactory.getLogger(QueryCaches.class);

	final private static int ARRAY_SIZE = 2*ClusterTraitsBase.getClusterArraySize();

	final private QueryCache[] cacheArray = new QueryCache[ARRAY_SIZE];

	public volatile boolean dirty = false;

	private int currentSize = 0;

	public boolean collecting = false;

	public boolean loading = false;

	AtomicInteger updates = new AtomicInteger(0);
	AtomicInteger size = new AtomicInteger(0);

	final QueryProcessor processor;
	final QuerySupport querySupport;
	final ClusteringSupportImpl clusteringSupport;
	final int threads;

	final private TreeMap<Long, QueryCacheBase> priorityQueue = new TreeMap<>();

	public QueryCaches(QueryProcessor processor, int threads) {
		this.processor = processor;
		this.querySupport = processor.querySupport;
		this.clusteringSupport = (ClusteringSupportImpl)processor.getSession().getService(ClusteringSupport.class);
		this.threads = threads;
		cacheArray[0] = new QueryCache(this, 0);
	}

	final public int r1(long id) {
		return (int)(id>>>32);
	}

	int cacheKey(int clusterKey, boolean user) {
		return (clusterKey << 1) + (user?1:0);
	}

	long cacheId(long clusterId, boolean user) {
		return (clusterId << 1) + (user?1:0);
	}

	boolean isUser(int cacheIdOrKey) {
		return (cacheIdOrKey&1) == 1;
	}

	synchronized QueryCache get(int r) {

		if(r < 0)
			return cacheArray[0];

		int clusterKey = ClusterTraitsBase.getClusterKeyFromResourceKeyNoThrow(r);
		return getByClusterCacheKey(cacheKey(clusterKey, false));

	}

	synchronized QueryCache getByClusterCacheKey(int cacheKey) {

		QueryCache cache = cacheArray[cacheKey];
		if(cache == null) {
			cache = new QueryCache(this, cacheKey);
			cacheArray[cacheKey] = cache;
			if(!loading) {
				int clusterKey = cacheKey >> 1;
				boolean immutable = clusterKey == 0 ? false : clusteringSupport.isImmutableClusterId(clusterKey);
				if(immutable) {
					File workspace = Platform.getLocation().toFile();
					File dir = new File(workspace, "queryData");
					if(dir.exists()) {
						long fileId = (clusteringSupport.clusterIdByClusterKey(clusterKey) << 1) + (isUser(cacheKey)?1:0);
						File f = new File(dir, fileId + ".queryData");
						if(f.exists()) {
							try {
								byte[] bytes = FileUtils.readFile(f);
								QueryDeserializerImpl qd = new QueryDeserializerImpl(processor, bytes);
								boolean immutable2 = qd.readHeaders(false);
								if(immutable2) {
									qd.readQueries();
								}
							} catch (IOException e) {
								LOGGER.error("Error while restoring queries");
							}
						}
					}
				}
			}
		}
		return cache;
	}

	QueryCache get(long id) {
		return get(r1(id));
	}

	QueryCache get(MultiRead<?> read) {
		return cacheArray[0];
	}

	QueryCache get(AsyncMultiRead<?> read) {
		return cacheArray[0];
	}

	QueryCache get(ExternalRead<?> read) {
		return cacheArray[0];
	}

	QueryCache get(Read<?> read) {
		if(read instanceof PersistentRead) {
			long cacheId = ((PersistentRead)read).cacheId(clusteringSupport);
			long clusterId = cacheId >> 1;
				if(clusterId == 0)
					return cacheArray[0];
				int clusterKey = clusteringSupport.clusterKeyByClusterId(clusterId);
				return getByClusterCacheKey(cacheKey(clusterKey, true));
		}
		return cacheArray[0];
	}

	QueryCache get(AsyncRead<?> read) {
		return cacheArray[0];
	}

	QueryCache get(String id) {
		return cacheArray[0];
	}

	void resetUpdates() {
		updates.set(0);
	}

	CacheCollectionResult allCachesForGC(CacheCollectionResult result) {
		for(int i=0;i<ARRAY_SIZE;i++) {
			QueryCache cache = cacheArray[i];
			if(cache != null)  {
				if(!cache.isImmutable())
					cache.allCaches(result);
			}
		}
		return result;
	}

	public Collection<CacheEntry<?>> getRootList() {
		ArrayList<CacheEntry<?>> result = new ArrayList<>();
		for(int i=0;i<ARRAY_SIZE;i++) {
			QueryCache cache = cacheArray[i];
			if(cache != null)
				result.addAll(cache.getRootList());
		}
		return result;
	}

	public int calculateCurrentSize() {
		int result = 0;
		for(int i=0;i<ARRAY_SIZE;i++) {
			QueryCache cache = cacheArray[i];
			if(cache != null)
				result += cache.calculateCurrentSize();
		}
		currentSize = result;
		return result;
	}

	public int getCurrentSize() {
		return currentSize;
	}

	public long getActivity() {
		return updates.get();
	}

	ExternalReadEntry getExternal(ExternalRead<?> request) {
		return cacheArray[0].externalReadEntryMap.get(request);
	}
	
	Collection<QueryCache> immutableCaches() {
		ArrayList<QueryCache> result = new ArrayList<>();
		for(int i=0;i<ARRAY_SIZE;i++) {
			QueryCache cache = cacheArray[i];
			if(cache != null) {
				if(cache.isImmutable())
					result.add(cache);
			}
		}
		return result;
	}
	
	Collection<QueryCache> mutableCaches() {
		ArrayList<QueryCache> result = new ArrayList<>();
		for(int i=0;i<ARRAY_SIZE;i++) {
			QueryCache cache = cacheArray[i];
			if(cache != null) {
				if(!cache.isImmutable())
					result.add(cache);
			}
		}
		return result;
	}

	public void registerPriorityForSwap(QueryCacheBase cache) {
		if(!cache.isImmutable())
			return;
		if(cache.storedPriority > 0)
			priorityQueue.remove(cache.storedPriority);
		if(cache.currentPriority > 0) {
			priorityQueue.put(cache.currentPriority, cache);
			cache.storedPriority = cache.currentPriority;
		}
	}

	public void dropAll(WriteGraphImpl graph) {
		// We drop everything but the special index 0
		for(int i=1;i<ARRAY_SIZE;i++) {
			cacheArray[i] = null;
		}
		// Invalidate all standard queries
		for(CacheEntry entry : cacheArray[0].getRootList()) {
			if(entry instanceof ExternalReadEntry)
				continue;
			processor.markForUpdate(graph, entry);
		}
	}
	
	public QueryCacheBase getSwapCandidate(int targetSize) {
		
		if(priorityQueue.size() > targetSize) {
			return priorityQueue.firstEntry().getValue();
		}

		return null;

	}

	/*
	 * Returns amount of queries persistent, -1 if not
	 */
	public int swapSingle(int targetSize) throws IOException {
		
		QueryCacheBase base = getSwapCandidate(targetSize);
		if(base == null)
			return -1;
		
		priorityQueue.remove(base.storedPriority);
		cacheArray[base.cacheKey] = null;
		
		return base.persist(processor);
		
	}
	
	public int swap(int targetSize) throws IOException {
		int count = 0;
		while(true) {
			int worked = swapSingle(targetSize);
			if(worked == -1)
				return count;
			count+=worked;
		}
	}

}
