/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.Arrays;

import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;

import gnu.trove.procedure.TIntProcedure;
import gnu.trove.set.hash.TIntHashSet;

public final class PrincipalTypes extends CollectionUnaryQuery {

	public PrincipalTypes(final int resource) {
		super(resource);
	}

	@Override
	final public void removeEntry(QueryProcessor provider) {
		QueryCache.remove(provider, this);
	}

	static class Ints {

		private TIntHashSet set = null;
		public int single = 0;

		public boolean add(int val) {
			if(single == val) return false;
			if(single == 0) {
				single = val;
				return true;
			}
			if(set == null) set = new TIntHashSet(4);
			set.add(val);
			return true;
		}

		public int size() {

			if(single == 0) return 0;
			if(set == null) return 1;
			return set.size() + 1;

		}

		public int[] toArray() {

			int[] result = Arrays.copyOf(set.toArray(), set.size() + 1);
			result[set.size()] = single;
			return result;

		}

		public void forEach(TIntProcedure proc) {
			proc.execute(single);
			if(set != null) set.forEach(proc);
		}

	}

	@Override
	public void compute(final ReadGraphImpl graph, final IntProcedure proc) throws DatabaseException {
		computeForEach(graph, id, this, proc);
	}
	
	public static Object computeForEach(final ReadGraphImpl graph, final int id, final PrincipalTypes entry, final IntProcedure procedure_) throws DatabaseException {

	    IntProcedure procedure = entry != null ? entry : procedure_;
	    
	    Object result = computeForEach2(graph, id, entry, procedure);
	    
	    if(entry != null) entry.performFromCache(graph, procedure_);
	    
	    return result;
	    
	}
	    
	public static Object computeForEach2(final ReadGraphImpl graph, final int id, final PrincipalTypes parent, final IntProcedure procedure) throws DatabaseException {

	    QueryProcessor provider = graph.processor;
		
		provider.querySupport.ensureLoaded(graph, id);
		assert(id != 0);

		int ret = provider.querySupport.getSingleInstance(id);
		if(ret > 0) {
			procedure.execute(graph, ret);
			procedure.finished(graph);
			return ret;
		}

		final int instanceOf = provider.getInstanceOf();
		final int inherits = provider.getInherits();
		final int subrelationOf = provider.getSubrelationOf();

		final Ints indirect = new Ints();
		final Ints material = new Ints();

		IntProcedure directProc = new IntProcedure() {

			@Override
			public void execute(ReadGraphImpl graph, int i) {
				material.add(i);
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
				procedure.exception(graph, t);
			}

			@Override
			public void finished(ReadGraphImpl graph) {
			}

		};

		IntProcedure indirectProc = new IntProcedure() {

			@Override
			public void execute(ReadGraphImpl graph, int i) {
				indirect.add(i);
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
				procedure.exception(graph, t);
			}

			@Override
			public void finished(ReadGraphImpl graph) {
			}

		};

		provider.querySupport.getObjects(graph, id, instanceOf, directProc);
		provider.querySupport.getObjects(graph, id, inherits, indirectProc);
		provider.querySupport.getObjects(graph, id, subrelationOf, indirectProc);

		if(indirect.size() == 0) {
			int size = material.size(); 
			if(size == 0) {
				procedure.finished(graph);
				return null;
			} else if(size == 1) {
				int single = material.single; 
				procedure.execute(graph, single);
				procedure.finished(graph);
				return single;
			} else {
				addPrincipalType(graph, new TIntHashSet(4), material.toArray(), 0, provider, parent, procedure);
				return null;
			}
		}

		indirect.forEach(new TIntProcedure() {

			int finishes = 0;
			
			@Override
			public boolean execute(final int arg0) {
				try {
					return execute0(arg0);
				} catch (DatabaseException e) {
					Logger.defaultLogError(e);
				}
				return false;
			}
			
			public boolean execute0(final int arg0) throws DatabaseException {

				// No self-loops!
				if(arg0 == id) {
					int current = (++finishes);
					if(current == indirect.size()) {
						int size = material.size(); 
						if(size == 0) {
							procedure.finished(graph);
							return true;
						} else if(size == 1) {
							int single = material.single; 
							procedure.execute(graph, single);
							procedure.finished(graph);
							return true;
						} else {
							addPrincipalType(graph, new TIntHashSet(4), material.toArray(), 0, provider, parent, procedure);
							return true;
						}
					}
					return true;
				}

				QueryCache.runnerPrincipalTypes(graph, arg0, parent, null, new IntProcedure() {

					@Override
					public void execute(ReadGraphImpl graph, int i) {
						synchronized(material) {
							material.add(i);
						}
					}

					@Override
					public void finished(ReadGraphImpl graph) throws DatabaseException {

						int current = (++finishes);
						if(current == indirect.size()) {
							int size = material.size(); 
							if(size == 0) {
								procedure.finished(graph);
								return;
							} else if(size == 1) {
								int single = material.single; 
								procedure.execute(graph, single);
								procedure.finished(graph);
								return;
							} else {
								addPrincipalType(graph, new TIntHashSet(4), material.toArray(), 0, provider, parent, procedure);
								return;
							}
						}

					}

					@Override
					public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
						procedure.exception(graph, t);
					}

				});

				return true;

			}

		});
		
		return null;

	}

	private static void finish(ReadGraphImpl graph, final TIntHashSet rejects, final int[] material, final IntProcedure proc) throws DatabaseException {

		for(int i : material) {
			if(!rejects.contains(i)) {
				proc.execute(graph, i);
			}
		}

		proc.finished(graph);

		return;

	}

	private static void addPrincipalType(final ReadGraphImpl graph, final TIntHashSet rejects, final int[] material, int index, final QueryProcessor provider, final PrincipalTypes parent, final IntProcedure proc) throws DatabaseException {

		if(index == material.length) { 
			finish(graph, rejects, material, proc);
			return;
		}

		int type = material[index++];
		while(rejects.contains(type)) {
			if(index == material.length) { 
				finish(graph, rejects, material, proc);
				return;
			}
			type = material[index++];
		}
		
		final int nextIndex = index;

		QueryCache.runnerSuperTypes(graph, type, parent, null, new InternalProcedure<IntSet>() {

			@Override
			public void execute(ReadGraphImpl graph, IntSet supers) throws DatabaseException {

				synchronized(rejects) {

					supers.forEach(new TIntProcedure() {

						@Override
						public boolean execute(int arg0) {
							rejects.add(arg0);
							return true;
						}

					});

				}

				addPrincipalType(graph, rejects, material, nextIndex, provider, parent, proc);

			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
				proc.exception(graph, t);
			}

		});

	}

	@Override
	public String toString() {
		return "PrincipalTypes[" + id + "]";
	}

}
