/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db.impl.query;

import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.request.RequestFlags;

final public class DirectObjects extends CollectionBinaryQuery<IntProcedure> implements IntProcedure {

    public DirectObjects(final int r1, final int r2) {
        super(r1, r2);
    }

    @Override
    public int type() {
        return RequestFlags.INVALIDATE;
    }

    @Override
    final public void removeEntry(QueryProcessor provider) {
        QueryCache.remove(provider, this);
    }

    // @Override
    public Object compute(ReadGraphImpl graph, final IntProcedure procedure) throws DatabaseException {
        computeForEach(graph, r1(), r2(), this, procedure);
        return getResult();
    }

    public static void computeForEach(ReadGraphImpl graph, int r1, int r2, final DirectObjects entry,
            final IntProcedure procedure_) throws DatabaseException {

        IntProcedure procedure = entry != null ? entry : procedure_;

        QueryProcessor processor = graph.processor;

        processor.querySupport.ensureLoaded(graph, r1, r2);

        processor.querySupport.getObjects(graph, r1, r2, procedure);

        procedure.finished(graph);

        if (entry != null)
            entry.performFromCache(graph, procedure_);

    }

    @Override
    public String toString() {
        return "DirectObjects[" + r1() + " - " + r2() + "]";
    }

    @Override
    public void setReady() {
        statusOrException = READY;
    }

    @Override
    public Object performFromCache(ReadGraphImpl graph, IntProcedure procedure) throws DatabaseException {

        assert (isReady());

        if (handleException(graph, procedure))
            return getResult();

        final IntArray value = (IntArray) getResult();
        if (value.data == null) {
            if (value.sizeOrData != IntArray.NO_DATA)
                procedure.execute(graph, value.sizeOrData);
        } else {
            for (int i = 0; i < value.sizeOrData; i++)
                procedure.execute(graph, value.data[i]);
        }

        procedure.finished(graph);

        return value;

    }

    @Override
    public void recompute(ReadGraphImpl graph) throws DatabaseException {

        compute(graph, new IntProcedure() {

            @Override
            public void finished(ReadGraphImpl graph) {
            }

            @Override
            public void exception(ReadGraphImpl graph, Throwable t) {
                throw new Error("Error in recompute.", t);
            }

            @Override
            public void execute(ReadGraphImpl graphd, int i) {
            }

        });

    }

    @Override
    boolean isImmutable(ReadGraphImpl graph) {
        return graph.processor.isImmutableForReading(r1());
    }

    @Override
    public void execute(ReadGraphImpl graph, int i) throws DatabaseException {
        IntArray value = (IntArray) getResult();
        value.add(i);
    }

    @Override
    public void finished(ReadGraphImpl graph) throws DatabaseException {
        setReady();
    }

    @Override
    public void exception(ReadGraphImpl graph, Throwable throwable) throws DatabaseException {
        except(throwable);
    }

}
