/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy  - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.Platform;
import org.osgi.framework.Bundle;
import org.osgi.framework.wiring.BundleWiring;
import org.simantics.db.ObjectResourceIdMap;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ClusterSupport;
import org.simantics.db.impl.ClusterTraitsBase;
import org.simantics.db.impl.ClusteringSupportImpl;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.impl.support.ResourceSupport;
import org.simantics.db.request.PersistentRead;
import org.simantics.db.request.QueryDeserializer;
import org.simantics.db.request.QueryFactory;
import org.simantics.db.request.QueryFactoryKey;
import org.simantics.db.request.Read;
import org.simantics.db.service.Bytes;
import org.simantics.db.service.ClusteringSupport;
import org.slf4j.LoggerFactory;

import gnu.trove.map.hash.TIntIntHashMap;

public class QueryDeserializerImpl implements QueryDeserializer {

	private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(QueryDeserializerImpl.class);

	QueryListening listening;
	ResourceSupport resourceSupport;
	ClusteringSupportImpl clusteringSupport;
	QueryCaches qc;
	QuerySupport qs;
	ClusterSupport cs;
	CacheEntry entry;

	private byte[] bytes;
	private int byteIndex;
	private boolean immutable;
	private long cacheId;

	private TIntIntHashMap clusterKeys = new TIntIntHashMap();
	private Map<Integer,QueryFactory> ids = new HashMap<Integer,QueryFactory>();

	public QueryDeserializerImpl(QueryProcessor qp, byte[] bytes) {
		this.listening = qp.listening;
		this.resourceSupport = qp.resourceSupport;
		this.clusteringSupport = (ClusteringSupportImpl)qp.getSession().getService(ClusteringSupport.class);
		this.qc = qp.caches;
		this.qs = qp.querySupport;
		this.cs = qs.getClusterSupport();;
		this.bytes = bytes;
	}

	public byte readByte() {
		return bytes[byteIndex++];
	}

	public boolean peekZero() {
		return bytes[byteIndex] == 0 && bytes[byteIndex+1] == 0;
	}

	public int readLE2() {
		int result = Bytes.readLE2(bytes, byteIndex);
		byteIndex += 2;
		return result;
	}

	public int readLE4() {
		int result = Bytes.readLE4(bytes, byteIndex);
		byteIndex += 4;
		return result;
	}

	public long readLE8() {
		long result = Bytes.readLE8(bytes, byteIndex);
		byteIndex += 8;
		return result;
	}

	public byte[] readBytes(int len) {
		byte[] result = Arrays.copyOfRange(bytes, byteIndex, byteIndex+len);
		byteIndex += len;
		return result;
	}

	public int registerImmutability(long clusterId, boolean immutable) {
		try {
			int clusterKey = cs.getClusterTable().getClusterKeyByUID(0, clusterId);
			cs.getClusterTable().markImmutable(clusterKey, immutable);
			return clusterKey;
		} catch (DatabaseException e) {
			e.printStackTrace();
		}
		return 0;
	}

	public boolean readHeaders(boolean skipImmutable) {

		cacheId = readLE8();

		long clusterId = cacheId >> 1;

		immutable = readByte() == 1;
		if(clusterId > 0)
			registerImmutability(clusterId, immutable);

		if(immutable && skipImmutable)
			return true;

		int idsSize = readLE4();
		for(int i=0;i<idsSize;i++) {
			int size = readLE4();
			QueryFactoryKey id = QueryFactoryKey.read(new String(readBytes(size)));
			int key = readLE4();
			try {
				Bundle bundle = Platform.getBundle(id.getBundle());
				BundleWiring bundleWiring = bundle.adapt(BundleWiring.class);
				ClassLoader classLoader = bundleWiring.getClassLoader();
				Class<QueryFactory> clazz = (Class<QueryFactory>)classLoader.loadClass(id.classId() + "Factory");
				QueryFactory qf = clazz.getDeclaredConstructor().newInstance(); 
				ids.put(key, qf);
			} catch (ClassNotFoundException e) {
				LOGGER.error("Error while resolving QueryFactory", e);
			} catch (InstantiationException e) {
				LOGGER.error("Error while resolving QueryFactory", e);
			} catch (IllegalAccessException e) {
				LOGGER.error("Error while resolving QueryFactory", e);
			} catch (IllegalArgumentException e) {
				LOGGER.error("Error while resolving QueryFactory", e);
			} catch (InvocationTargetException e) {
				LOGGER.error("Error while resolving QueryFactory", e);
			} catch (NoSuchMethodException e) {
				LOGGER.error("Error while resolving QueryFactory", e);
			} catch (SecurityException e) {
				LOGGER.error("Error while resolving QueryFactory", e);
			}
		}
		int clusterKeysSize = readLE4();
		for(int i=0;i<clusterKeysSize;i++) {
			long referencedClusterId = readLE8();
			int key = readLE4();
			boolean immu = readByte() == 1;
			int actualClusterKey = registerImmutability(referencedClusterId, immu);
			clusterKeys.put(key, actualClusterKey);
		}

		return immutable;

	}

	public QueryFactory readFactory() {
		int key = readLE4();
		return ids.get(key);
	}

	private static int reads0 = 0;
	private static int reads1 = 0;

	public int readQueries() {
		int count = readLE4();
		for(int i=0;i<count;i++) {
			QueryFactory qf = readFactory();
			try {
				qf.read(this);
			} catch (DatabaseException e) {
				LOGGER.error("Error while deserializing query", e);
			} catch (Throwable t) {
				LOGGER.error("Failure to deserialize queries", t);
				throw new IllegalStateException("Failure to deserialize queries");
			}
		}

		if(immutable) {
			if((cacheId & 1) == 1) {
				reads1++;
			} else {
				reads0++;
				//new Exception("[" + reads1 + " " + reads0 + "] Loaded primitive queries for immutable cache " + cacheId + " " + count).printStackTrace();
			}
		}

		return count;

	}

	public Resource readResource() throws DatabaseException {
		int id = readResourceI();
		if(id == 0)
			return null;
		return new ResourceImpl(resourceSupport, id);
	}

	public Statement readStatement() throws DatabaseException {
		int s = readResourceI();
		int p = readResourceI();
		int o = readResourceI();
		return qs.getStatement(s,p,o);
	}

	public int readResourceI() throws DatabaseException {
		if(peekZero()) {
			return readLE2();
		}
		int key = readLE4();
		if(key <= 0)
			return key;
		int clusterKey = ClusterTraitsBase.getClusterKeyFromResourceKey(key);
		int actualClusterKey = clusterKeys.get(clusterKey);
		return ClusterTraitsBase.createResourceKey(actualClusterKey, ClusterTraitsBase.getResourceIndexFromResourceKey(key));
	}

	public byte[] readByteArray() {
		int len = readLE4();
		if(len == -1)
			return null;
		return readBytes(len);
	}

	public String readString() {
		return new String(readByteArray());
	}

	public ObjectResourceIdMap<String> createChildMap() {
		return qs.createChildMap();
	}

	AssertedPredicates readAssertedPredicates() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreateAssertedPredicates(r);
	}

	AssertedStatements readAssertedStatements() throws DatabaseException {
		int r1 = readResourceI();
		int r2 = readResourceI();
		return qc.get(r1).getOrCreateAssertedStatements(r1, r2);
	}

	ChildMap readChildMap() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreateChildMap(r);
	}

	DirectObjects readDirectObjects() throws DatabaseException {
		int r1 = readResourceI();
		int r2 = readResourceI();
		return qc.get(r1).getOrCreateDirectObjects(r1, r2);
	}

	DirectPredicates readDirectPredicates() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreateDirectPredicates(r);
	}

	Objects readObjects() throws DatabaseException {
		int r1 = readResourceI();
		int r2 = readResourceI();
		return qc.get(r1).getOrCreateObjects(r1, r2);
	}

	OrderedSet readOrderedSet() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreateOrderedSet(r);
	}

	Predicates readPredicates() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreatePredicates(r);
	}

	PrincipalTypes readPrincipalTypes() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreatePrincipalTypes(r);
	}

	RelationInfoQuery readRelationInfoQuery() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreateRelationInfoQuery(r);
	}

	Statements readStatements() throws DatabaseException {
		int r1 = readResourceI();
		int r2 = readResourceI();
		return qc.get(r1).getOrCreateStatements(r1, r2);
	}

	SuperRelations readSuperRelations() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreateSuperRelations(r);
	}

	SuperTypes readSuperTypes() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreateSuperTypes(r);
	}

	TypeHierarchy readTypeHierarchy() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreateTypeHierarchy(r);
	}

	Types readTypes() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreateTypes(r);
	}

	URIToResource readURIToResource() throws DatabaseException {
		String s = readString();
		return qc.get(s).getOrCreateURIToResource(s);
	}

	ValueQuery readValueQuery() throws DatabaseException {
		int r = readResourceI();
		return qc.get(r).getOrCreateValueQuery(r);
	}

	public void setEntry(CacheEntry entry) {
		this.entry = entry;
	}

	public CacheEntry getEntry() {
		return entry;
	}

	@Override
	public <K,V> void store(K key, V value) {
		if(key instanceof Read) {
			Read r = (Read)key;
			setEntry(qc.get(r).getOrCreateReadEntry(r));
		} else if (key instanceof CacheEntry) {
			setEntry((CacheEntry)key);
		} else {
			throw new IllegalStateException("Illegal key " + key);
		}
		if(value instanceof Throwable) {
			entry.except((Throwable)value);
		} else {
			entry.setResult(value);
			entry.setReady();
		}
	}

	public <K> void parent(K key) {

		// This cluster is immutable and its queries do not have parents
		if(immutable)
			return;

		if(key instanceof PersistentRead) {

			PersistentRead r = (PersistentRead)key;
			long cacheId = r.cacheId(clusteringSupport);
			long clusterId = cacheId >> 1;
			boolean immutable = clusteringSupport.isImmutableClusterId(clusterId);
			if(immutable)
				return;

			ReadEntry entry = qc.get(r).getOrCreateReadEntry(r);
			listening.addParent(getEntry(), entry);

		} else if (key instanceof CacheEntry) {
			listening.addParent(getEntry(), (CacheEntry)key);
		} else {
			throw new IllegalStateException("Illegal key " + key);
		}
	}

}
