/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.common.exception.DebugException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;

final public class OrderedSet extends CollectionUnaryQuery {

	public OrderedSet(final int r) {
		super(r);
	}

	@Override
	final public void removeEntry(QueryProcessor provider) {
		QueryCache.remove(provider, this);
	}

	private static int nextElement(ReadGraphImpl graph, int current, int orderedSet, OrderedSet parent, final IntProcedure procedure) throws DatabaseException {

		QueryProcessor processor = graph.processor;

		processor.querySupport.ensureLoaded(graph, current);

		AtomicInteger res = new AtomicInteger(0);

		processor.querySupport.getObjects(graph, current, orderedSet, new IntProcedure() {

			@Override
			public void execute(ReadGraphImpl graph, int i) throws DatabaseException {
				if(i != orderedSet) {
					procedure.execute(graph, i);
				}
				res.set(i);
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
				if(DebugException.DEBUG) new DebugException(t).printStackTrace();
				procedure.exception(graph, t);
			}

			@Override
			public void finished(ReadGraphImpl graph) {
			}

		});

		if(res.get() == orderedSet) {
			procedure.finished(graph);
		}

		return res.get();

	}

	@Override
	public void compute(ReadGraphImpl graph, final IntProcedure procedure) throws DatabaseException {
		computeForEach(graph, id, this, procedure);
	}

	static void computeForEach(ReadGraphImpl graph, int orderedSet, final OrderedSet entry, final IntProcedure procedure_) throws DatabaseException {

		IntProcedure procedure = entry != null ? entry : procedure_;

		int current = nextElement(graph, orderedSet, orderedSet, entry, procedure);
		while(current != orderedSet) {
			current = nextElement(graph, current, orderedSet, entry, procedure);
		}

		if(entry != null) entry.performFromCache(graph, procedure_);

	}

	@Override
	public String toString() {
		return "OrderedSet[" + id + "]";
	}

}
